﻿//*************************************\\
//*
//*  :: H T T P     R O T A T O R ::
//*
//*************************************\\
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Linq;
using PacMap.General;
using PacMap.Anonymize.Transactions;

namespace PacMap.Anonymize.HTTPRotator
{
    /// <summary>
    /// HTR - HTTP Text Rotator
    /// </summary>
    public class Rotator
    {

        /// <summary>
        /// Constructs HTR - HTTP Text Rotator
        /// </summary>
        public Rotator()
        {
            collection = new List<RotatorItem>();
        }


        /****************************************************************************************/
        // :: DATA FIELDs ::
        IList<RotatorItem> collection = null;
        private string lastError = null;
        private bool isCorrect = true;

        /****************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Get Rotator Item
        /// </summary>
        public IEnumerable<RotatorItem> Items
        {
            get
            {
                return collection;
            }
        }

        /// <summary>
        /// Get LastError (if any)
        /// </summary>
        public string LastError
        {
            get
            {
                string result = "";
                if (lastError != null)
                    result = lastError;
                return result;
            }
        }

        /// <summary>
        /// HTTP Rotator is in Correct State
        /// </summary>
        public bool IsCorrect
        {
            get
            {
                return isCorrect;
            }
        }
        #endregion

        /****************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Add new HTTP Rotator Item into HTTP Rotator (manually) - using just for defining defaults by programmer not user
        /// </summary>
        /// <param name="newItem">new HTTP Rotator Item</param>
        public void Add(RotatorItem newItem)
        {
            collection.Add(newItem);
        }


        /// <summary>
        /// Make Single Rotation (=Do the Action on given string collection)
        /// </summary>
        /// <param name="inputCollection">HTTP header collection</param>
        /// <returns>return new HTTP header collection</returns>
        public IList<string> Rotate(string[] inputCollection)
        {
            IList<string> newCollection = inputCollection.ToList();

            foreach (var hrtitem in Items) // for every single rotator item in HTTP Rotator do...
            {
                string item = newCollection.ToArray().ParseHeaderIntelligence(hrtitem.CollectionTarget); // rechoose certain HTTP header line, which corresponds to HTTP parameter (e.g. "HOST=", "Cookie:=", etc.)

                if (item != "")
                {
                    bool willReplace = false;
                    if (hrtitem.ReplaceAlways) // action is going to proceed everytime, for any inputed string => no filtering supported
                        willReplace = true;
                    if (hrtitem.ReplaceOnlyIfFindedExactExpression) // action is going to proceed in one condition : inputed string is exactly equal to name of HTTP parameter => it means filtering enabled
                    {
                        if (item == hrtitem.WhatToBeReplaced)
                            willReplace = true;
                    }
                    else // action is going to proceed when inputed string is substring of HTTP parameter name
                    {
                        if (item.Contains(hrtitem.WhatToBeReplaced))
                            willReplace = true;
                    }


                    string innerResult = hrtitem.WhatToBeReplaced; // the text which may be replaced (or may not)


                    if (willReplace) // if action is going to proceed for sure, then...
                    {
                        if (!hrtitem.ContainsTransaction) // if rotator item is simple
                        {
                            innerResult = hrtitem.ReplacingContent;
                        }
                        else // if rotator item is "overrided" into any transaction
                        {
                            AnonValueTransaction valueAction = hrtitem.Transaction as AnonValueTransaction;
                            AnonDateTransaction dateAction = hrtitem.Transaction as AnonDateTransaction;
                            AnonStringTransaction stringAction = hrtitem.Transaction as AnonStringTransaction;
                            if (valueAction != null)
                            {
                                innerResult = item;
                                int original;
                                if (int.TryParse(innerResult, out original))
                                {
                                    int export = valueAction.Run(original);
                                    innerResult = export.ToString();
                                }
                            }

                            if (dateAction != null)
                            {
                                innerResult = dateAction.Run(item);
                            }

                            if (stringAction != null)
                            {
                                innerResult = stringAction.Run(item);
                            }
                        }
                    }


                    newCollection = newCollection.ToArray().SaveHeaderIntelligence(hrtitem.CollectionTarget, innerResult, true);
                }
                else // if there is no target in headerCollection
                {
                }
            }



            return newCollection;
        }


        /// <summary>
        /// Import XML Settings into the NEW HTTP Rotator HTR
        /// </summary>
        /// <param name="element">XML element (name = "rotator")</param>
        /// <returns>new HTR Unit</returns>
        public static Rotator Import(XElement element)
        {
            Rotator rotator = new Rotator();
            bool anyError = false;

            IEnumerable<XElement> items = element.Elements("item");
            if (items != null)
            {
                foreach (var item in items)
                {
                    string name = (string)item.Attribute("name");
                    if (name == "rotator-item")
                    {
                        string action = (string)item.Attribute("rotate");
                        if (action != null)
                        {
                            RotatorItem rotitem;
                            if (RotatorItem.TryParse(action, out rotitem))
                            {
                                rotator.collection.Add(rotitem);
                            }
                            else
                            {
                                rotator.lastError = String.Format("bad or unknown rotation \"{0}\"", action);
                            }
                        }
                        else
                        {
                            rotator.lastError = "\"rotate\" tag expected";
                        }
                    }
                }


            }

            if (rotator.lastError != null)
                anyError = true;

            rotator.isCorrect = !anyError;
            return rotator;
        }


        /// <summary>
        /// Export all Rotator Items into XML Element
        /// </summary>
        /// <returns>XML Element</returns>
        public XElement Export()
        {
            XElement result = new XElement("rotator");

            if (collection.Count != 0)
            {
                foreach (var item in collection)
                {
                    XElement itemXML = new XElement("item");
                    itemXML.SetAttributeValue("name", "rotator-item");
                    itemXML.SetAttributeValue("rotate", item.ToString());
                    result.Add(itemXML);
                }

            }


            return result;
        }


        /****************************************************************************************/
        // :: FUNCTIONs ::


    }
}
