﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace PacMap.General.Masks
{
    /// <summary>
    /// USER TCP/UDP Port Address MASK Class
    /// </summary>
    public class MaskPorts : Mask
    {

        /// <summary>
        /// Construct Empty TCP/UDP Port Address Mask
        /// </summary>
        public MaskPorts()
            : base()
        {
        }

        /// <summary>
        /// Construct User TCP/UDP Port Address Mask
        /// </summary>
        /// <param name="byteLength">total mask length</param>
        /// <param name="maskType">mask type</param>
        /// <param name="maskedString">mask string ('true' - original, 'false' - anonmapping)</param>
        public MaskPorts(int byteLength, MaskEnum maskType, string maskedString)
            : base(byteLength, maskType, maskedString)
        {
        }


        /// <summary>
        /// Construct User TCP/UDP Port Address Mask from simple texted string (mandatory form: "XX---") ('X' - original, '-' anonmapping)
        /// </summary>
        /// <param name="input">mask string</param>
        public MaskPorts(string input)
        {
            if (input == "none")
            {
                isEmpty = true;
                isCorrect = true;
                return;
            }

            isEmpty = false;
            if (input.Length == 5)
            {
                if ((input.Contains('X')) || (input.Contains('-')))
                {
                    length = 2;
                    mask = new byte[length];

                    int item = 0;
                    for (int k = 1; k <= 5; k++)
                    {
                        if (input[k - 1] == 'X')
                        {
                            item = item | (0x01 << (5 - k));
                        }
                    }
                    mask[0] = (byte)item;


                    isSpecialized = true;
                    isCorrect = true;
                }
                else
                {
                    isCorrect = false;
                    lastError = "Mask string must contain \"X\" or \"-\"";
                }
            }
            else
            {
                isCorrect = false;
                lastError = "Mask string has to be 5 character long";
            }
        }

        /**********************************************************************************************/
        // :: DATA FIELDs ::
        private bool isSpecialized = false;

        /**********************************************************************************************/
        // :: PROPERTIEs ::

        /**********************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Proceed Mask on Input Item
        /// </summary>
        /// <param name="original">original data</param>
        /// <param name="anonmapped">anonmapped data</param>
        /// <returns>result string</returns>
        public override byte[] Proceed(byte[] original, byte[] anonmapped)
        {
            if ((IsEmpty) || (original.Length != anonmapped.Length) || (original.Length != mask.Length))
                return anonmapped;

            if (!isSpecialized)
                return base.Proceed(original, anonmapped);
            else
            {
                byte[] result = new byte[length];

                string originalString = String.Format("{0,00000}", original.ConvertToPortNumber());
                string anonmappedString = String.Format("{0,00000}", anonmapped.ConvertToPortNumber());
                string preresult = "";

                for (int k = 1; k <= 5; k++)
                {
                    if (((mask[0] & (0x01 << (5 - k))) >> (5 - k)) == 0x01)
                    {
                        preresult += originalString[k - 1];
                    }
                    else
                    {
                        preresult += anonmappedString[k - 1];
                    }
                }

                int value;
                int.TryParse(preresult, out value);

                result = value.ConvertPortNumberToBytesArray();

                return result;
            }
        }

        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>string result</returns>
        public override string ToString()
        {
            if (!IsCorrect)
                return "badmask";
            if (IsEmpty)
                return "none";

            string result = "";
            if (!isSpecialized)
                result = base.ToString();
            else
            {
                string item = "";
                for (int k = 1; k <= 5; k++)
                {
                    if (((mask[0] & (0x01 << (5 - k))) >> (5 - k)) == 0x01)
                        item += "X";
                    else
                        item += "-";
                }

                result = item;
            }
            return result;
        }

        /// <summary>
        /// Create New Mask from String
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>completed mask</returns>
        public static MaskPorts Parse(string input)
        {
            MaskPorts result = null;
            int byteLength;
            MaskEnum type;
            string maskedString;
            bool isCorrect = ParseBinHex(input, out byteLength, out type, out maskedString);

            if (!isCorrect)
            {
                result = new MaskPorts(input);
                isCorrect = result.IsCorrect;
            }
            else
            {
                result = new MaskPorts(byteLength, type, maskedString);
            }

            //if (isCorrect)
                return result;
            //else
            //    return new MaskPorts();
        }


    }
}
