﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.ComponentModel;
using PacMap.General;
using PacMap.Tracer;
using PacMap.Packet.Exceptions;
using PacMap.Anonymize.Templates;
using PacMap.Additionals;
using PacMap.Additionals.Exporting;
using System.Drawing;

namespace PacMap.Packet
{
    /// <summary>
    /// Packet Packager Manager
    /// </summary>
    public class PacketPackager : IPacketPackager
    {
        /// <summary>
        /// Constructs Empty Packet Packager Manager
        /// </summary>
        public PacketPackager()
        {
        }


        /********************************************************************/
        // :: DATA FIELDs ::
        private bool isEmpty = true;
        private bool isCorrect = true;
        private IList<Packet> packets = null;
        private long totalLength = 0;
        private string originalName = "";
        private byte[] header = null;
        private AppStatus innerState = AppStatus.Ready;

        /// <summary>
        /// Unique Communication Profile ((Information)) Collection
        /// </summary>
        private IDictionary<KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>>, object> uniqueCommunications = null;

        /// <summary>
        /// Dictionary - Given (Set Up) Template for every single Packet
        /// </summary>
        private IDictionary<Packet, Template> templateCollection = null;

        
        /********************************************************************/
        // :: PROPERTIEs ::
        public event ProgressChangedEventHandler ProgressChanged; 

        #region PROPERTIEs
        /// <summary>
        /// Get Packets Count
        /// </summary>
        public int Count
        {
            get
            {
                return packets.Count;
            }
        }

        /// <summary>
        /// Get PacketPackager State
        /// </summary>
        public AppStatus InnerState
        {
            get
            {
                return innerState;
            }
        }

        /// <summary>
        /// Get Packet Collection from PacketPackager object
        /// </summary>
        public IEnumerable<Packet> Items
        {
            get
            {
                return packets;
            }
        }

        /// <summary>
        /// Get Correct State of PacketPackager
        /// </summary>
        public bool IsCorrect
        {
            get
            {
                return ((isCorrect) && (!isEmpty));
            }
        }

        /// <summary>
        /// Get Packet Packager Name
        /// </summary>
        public string OriginalName
        {
            get
            {
                return originalName;
            }
        }

        /// <summary>
        /// Get Packet Packager Size
        /// </summary>
        public long Size
        {
            get
            {
                return totalLength;
            }
        }
        #endregion



        /********************************************************************/
        // :: PUBLIC METHODs ::



        private void ExportTraceFileContent(TextWriter writer, bool showOnlySimple, bool isPrinting)
        {
            if (!IsCorrect)
                throw new PacketPackagerNotCorrectException("Packet Packager is not in correct state!");

            int right = Program.ConsoleMaxX - 3;
            Action allignRightAndLeft = new Action(() =>
            {
                if (!isPrinting)
                {
                    Console.CursorLeft = right;
                    ConsoleGraphics.ColoredWrite("$08==\r== ");
                }
            });

            #region Writing down the Basic Info
            {
                GeneralExt.MakeConsoleLine(isPrinting, true, writer);

                string defaultSeparator = " ";
                if (AppSettings.FileFormat == FileFormat.ODT)
                    defaultSeparator = "\t";
                if (AppSettings.FileFormat == FileFormat.CSV)
                    defaultSeparator = ";";


                #region Writing down Basic Info
                {
                    IList<string> basicInfo = new List<string>();
                    basicInfo.Add(String.Format("File:{0}$07{1}", defaultSeparator, originalName));
                    basicInfo.Add(String.Format("Size:{0}$07{1}", defaultSeparator, totalLength.ByteSize()));
                    basicInfo.Add(String.Format("Total Packets:{0}$07{1}", defaultSeparator, packets.Count));

                    if (!isPrinting)
                    {
                        for (int i = 1; i <= 3; i++)
                        {
                            allignRightAndLeft(); // only for stdout (no printing)
                            ConsoleGraphics.ColoredWriteLine(basicInfo[i - 1]);
                        }
                    }
                    else
                    {
                        for (int i = 1; i <= 3; i++)
                        {
                            writer.WriteLine(basicInfo[i - 1].CleanColors());
                        }
                    }
                }
                #endregion

                if ((showOnlySimple) && (!isPrinting))
                    GeneralExt.MakeConsoleLine();

                #region Continuing writing down Basic Info
                {
                    string outputUniqueCommunications = String.Format("Unique Communications:{0}$07{1}", defaultSeparator, uniqueCommunications.Count);
                    if (!isPrinting)
                    {
                        allignRightAndLeft();
                        ConsoleGraphics.ColoredWriteLine(outputUniqueCommunications);
                    }
                    else
                    {
                        writer.WriteLine(outputUniqueCommunications.CleanColors());
                        writer.WriteLine("---");
                    }
                }
                #endregion

            }
            #endregion

            #region Writing down the Complete Chart
            if (showOnlySimple)
            {

                allignRightAndLeft();
                if (!isPrinting)
                    writer.WriteLine();

                //
                // Writing down the Chart Content
                //

                #region Prepare Data for Table
                IList<string> table_localip = new List<string>();
                IList<string> table_remoteip = new List<string>();
                IList<string> table_localport = new List<string>();
                IList<string> table_remoteport = new List<string>();
                IList<string> table_teimstring = new List<string>();
                int index = 0;
                foreach (var viacom in uniqueCommunications)
                {
                    index++;
                    string empty = "";
                    for (int a = 1; a <= 15; a++)
                        empty += " ";

                    string localIP = viacom.Key.Key.Key;    // get local IP from viacom tree structured object "viacom"
                    string remoteIP = viacom.Key.Key.Value; // get ...
                    int localPort = viacom.Key.Value.Key;
                    int remotePort = viacom.Key.Value.Value;

                    string itemsString = ((IList<Packet>)viacom.Value).Count.ToString();

                    table_localip.Add(localIP);
                    table_remoteip.Add(remoteIP);
                    table_localport.Add(localPort.ToString());
                    table_remoteport.Add(remotePort.ToString());
                    table_teimstring.Add(itemsString);
                }
                #endregion

                Table table = new ArrowTable();
                table.AddColumn("Local address", table_localip);
                table.AddColumn("Remote address", table_remoteip);
                table.AddColumn("Local port", table_localport);
                table.AddColumn("Remote port", table_remoteport);
                table.AddColumn("Packets", table_teimstring);
                table.MaxWidth = Program.ConsoleMaxX - 1;

                table.ChangeColumnWidth(3, 0.13d); // set 3rd width to 13%
                table.ChangeColumnWidth(4, 0.14d); // ---- || ----  to 14%
                table.ChangeColumnWidth(5, 0.12d); // ---- || ----  to 12%

                if (!isPrinting)
                    table.Print();
                else
                {
                    table.Print(AppSettings.FileFormat, writer);
                }

            }
            #endregion
        }
        

        private void ExportSimplePacketInfo(TextWriter writer, bool isPrinting)
        {
            if (!IsCorrect)
                throw new PacketPackagerNotCorrectException("Packet Packager is not in correct state!");

            ExportTraceFileContent(writer, false, isPrinting);

            #region Prepare Data for Table
            IList<string> table_no = new List<string>();
            IList<string> table_source = new List<string>();
            IList<string> table_destination = new List<string>();
            IList<string> table_protocol = new List<string>();
            IList<string> table_info = new List<string>();

            int a = 0;
            foreach (var item in packets)
            {
                a++;
                string source = item.IPSource;
                string destination = item.IPDestination;
                string protocol = item.Protocol;
                string info = item.Info;
                                
                if (info == "")
                {
                    try
                    {
                        info = (item.PortInfo.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries)).ToList()[0];
                        info = "$08" + info;
                    }
                    catch
                    {
                    }
                }

                table_no.Add(a.ToString());
                table_source.Add(source);
                table_destination.Add(destination);
                table_protocol.Add(protocol);
                table_info.Add(info);
            }
            #endregion


            Table table = new Table();
            table.AddColumn("No", table_no);
            table.AddColumn("Source", table_source);
            table.AddColumn("Destination", table_destination);
            table.AddColumn("Protocol", table_protocol);
            table.AddColumn("Information", table_info);

            table.MaxWidth = Program.ConsoleMaxX - 1;

            table.ChangeColumnWidth(1, 0.10d); // set 1st column width to 10%
            table.ChangeColumnWidth(2, 0.22d); // set 2nd column width to 22%
            table.ChangeColumnWidth(3, 0.22d); // set 3rd column width to 22%
            table.ChangeColumnWidth(4, 0.10d); // set 4th column width to 10%
            // 5th column width is variable and it is (100% - sum_of(first 4 column))

            if (!isPrinting)
                table.Print(); // Simply show whole table
            else
                table.Print(AppSettings.FileFormat, writer);
        }






        public bool PrintTraceFileContent(FileInfo input)
        {
            bool bResult = true;
            try
            {
                using (TextWriter writer = new StreamWriter(input.FullName))
                {
                    ExportTraceFileContent(writer, true, true);
                }
            }
            catch
            {
                bResult = false;
            }
            
            return bResult;
        }

        public bool PrintSimplePacketInfo(FileInfo input)
        {   
            bool bResult = true;
            try
            {
                using (TextWriter writer = new StreamWriter(input.FullName))
                {
                    ExportSimplePacketInfo(writer, true);
                }
            }
            catch
            {
                bResult = false;
            }

            
            return bResult;
        }




        public void ShowTraceFileContent()
        {
            ExportTraceFileContent(Console.Out, true, false);
        }

        public void ShowSimplePacketInfo()
        {
            ExportSimplePacketInfo(Console.Out, false);
        }

        public void ShowAdvancedPacketInfo()
        {
            PacketBrowser browser = new PacketBrowser(this);
            browser.Start();
        }



        public bool Filter(Point span, out IList<Packet> newList)
        {
            bool bResult = false;
            int left = span.X;
            int right = span.Y;

            newList = null;

            if ((left >= 1) && (right <= packets.Count))
            {
                bResult = true;
                newList = new List<Packet>();

                for (int index = 1; index <= packets.Count; index++)
                {
                    Packet packet = packets[index - 1];

                    if ((left <= index) && (index <= right))
                        newList.Add(packet);
                }
            }

            return bResult;
        }


        
        public bool Anonmap()
        {   

            if (!IsCorrect)
                throw new PacketPackagerNotCorrectException("Packet Packager is not in correct state!");

            bool bresult = false;
            if (innerState != AppStatus.Ready)
                throw new PacketPackagerCrossThreadsException("Packaget Packeger is already running operation!");
                
            this.innerState = AppStatus.AnonMappings;
            int index = 0;
            foreach (var packet in packets)
            {
                index++;
                packet.Anonymize(templateCollection[packet]);
                OnProgressChanged(index, packets.Count, 0, 100); // Reports Progress into "ProgressChanged" event
            }
            this.innerState = AppStatus.Ready;
            bresult = true;
            return bresult;
        }

        
        public bool Save(string filename)
        {
            if (!IsCorrect)
                throw new PacketPackagerNotCorrectException("Packet Packager is not in correct state!");

            bool bresult = false;
            if (innerState != AppStatus.Ready)
                throw new PacketPackagerCrossThreadsException("Packaget Packeger is already running operation!");

            this.innerState = AppStatus.SavingFile;
            using (FileStream stream = new FileStream(filename, FileMode.Create))
            {
                stream.Position = 0;
                stream.Write(header, 0, header.Length);

                int index = 0;
                foreach (var packet in packets)
                {
                    index++;
                    byte[] output = packet.Save();
                    stream.Write(output, 0, output.Length);
                    OnProgressChanged(index, packets.Count, 0, 100); // Reports Progress into "ProgressChanged" event
                }
            }
            this.innerState = AppStatus.Ready;
            bresult = true;
            return bresult;
        }

        
        public bool LoadFile(string filename)
        {
            FileInfo fileInfo = new FileInfo(filename);
            if (fileInfo.Exists)
                return LoadFile(fileInfo);
            else
            {
                isCorrect = false;
                return false;
            }
        }

        
        public bool LoadFile(FileInfo filename)
        {
            bool bresult = false;
            if (!isEmpty) // if was already loaded
                throw new PacketPackagerNotEmptyException("Packet Packager is not empty for additional loading file!");
            if (innerState != AppStatus.Ready)
                throw new PacketPackagerCrossThreadsException("Packaget Packeger is already running operation!");


            this.innerState = AppStatus.ReadingFile;
            isEmpty = false;
            isCorrect = false;
            this.originalName = filename.Name;

            FileStream stream = File.OpenRead(filename.FullName); // get stream from filename

            stream.Position = 0;
            this.totalLength = stream.Length;

            byte[] pcapheader = new byte[24]; // initiate reading first header bytes
            stream.Read(pcapheader, 0, 24);
            if (!CheckFileHeader(pcapheader)) // verifying for PCAP header (corrent PCAP filename)
            {
                isCorrect = false;
                return bresult;
            }

            this.header = pcapheader; // filename size

            int length = (int)filename.Length;


            //
            // [1st/3] Get Single Packet Positions in PCAP File
            //
            IList<int> packet__positions = new List<int>();
            bool loop = true;
            while (loop)
            {
                byte[] packetInfoBuffer = new byte[16]; // every packet begins with first 16bytes (called frame) - cotains: 1) Captured Epoch Time; 2) Original Packet Size & 3) Captured Packet Size
                int iresult = 0;
                iresult = stream.Read(packetInfoBuffer, 0, 16); // read those bytes
                if (iresult > 0)                                // if it is possible -->
                {
                    packet__positions.Add((int)stream.Position - 16); // --> record the starting position of single one packet

                    int word = (packetInfoBuffer[9] << 8) | (packetInfoBuffer[8]); // get Captured Packet Size,
                    stream.Position += word;                                       // then set stream position on the end of this single packet
                }
                else                                            // if it is NOT possible -->
                    loop = false;                               // break the Loop

                this.OnProgressChanged((int)stream.Position, stream.Length, 0, 20); // Report Progress To "ProgressChanged" event (0 -> 20%)
            }



            //
            // [2nd/3] Get Single Packet Content (in bytes array) from PCAP File
            //
            IList<Stream> packets__binary = new List<Stream>();
            for (int i = 0; i <= packet__positions.Count - 2; i++)          // for every packet position except the last one -->
            {
                int size = packet__positions[i + 1] - packet__positions[i];
                byte[] buffer = new byte[size];
                stream.Position = packet__positions[i];
                stream.Read(buffer, 0, size);                               // --> read all the data (make stream)

                Stream result = new MemoryStream(buffer);
                packets__binary.Add(result);                                // & input packet stream in collection

                this.OnProgressChanged(i + 1, packet__positions.Count, 20, 50); // Report Progress To "ProgressChanged" event (20 -> 50%)
            }
            {
                int size = length - packet__positions.Last();               // Take the last position -->
                byte[] buffer = new byte[size];
                stream.Position = packet__positions.Last();                 // --> and read the last data in filename
                stream.Read(buffer, 0, size);

                Stream result = new MemoryStream(buffer);
                packets__binary.Add(result);                                // add it into the collection as well
            }



            //
            // [3rd/3] Get Single Packet (ready to operate) from PCAP File
            //
            this.packets = new List<Packet>();
            int index = 0;
            foreach (var item in packets__binary)   // For every stream packet in PCAP file -->
            {
                index++;
                Packet result = new Packet(item);   // --> Initialize Packet class constructor with stream parsing &
                this.packets.Add(result);       // & add to the main packet packager collection

                this.OnProgressChanged(index, packets__binary.Count, 50, 100); // Report Progress To "ProgressChanged" event (50 -> 100%)
            }

            stream.Dispose(); // close input filename
            this.innerState = AppStatus.Ready;
            isCorrect = true;
            bresult = true;

            CheckForUniqueCommunications();
            return bresult;
        }


        /********************************************************************/
        // :: PRIVATE HELPING FUNCTIONs ::
        /// <summary>
        /// Signal "ProgressChanged" event with special percentage
        /// </summary>
        /// <param name="percentage">percentage (0 - 100)</param>
        private void OnProgressChanged(int percentage)
        {
            ProgressChangedEventHandler handler = ProgressChanged;
            if (handler != null)
                handler(this, new ProgressChangedEventArgs(percentage, innerState));
        }

        /// <summary>
        /// Signal "ProgressChanged" event with advanced percentage value
        /// </summary>
        /// <param name="actualVariable">actual variable position</param>
        /// <param name="allVariables">all variables count</param>
        /// <param name="leftPercentageEdge">left percentage edge (0 - 100)</param>
        /// <param name="rightPercentageEdge">right percentage edge (0 - 100); (right edge must be always greater or equal than left edge)</param>
        private void OnProgressChanged(int actualVariable, long allVariables, int leftPercentageEdge, int rightPercentageEdge)
        {
            double ratio = ((double)actualVariable / allVariables);
            int value = leftPercentageEdge + (int)((ratio * (double)(rightPercentageEdge - leftPercentageEdge)));
            OnProgressChanged(value);
        }


        /// <summary>
        /// Check PCAP File Header
        /// </summary>
        /// <param name="input">first 24bytes in header</param>
        /// <returns>TRUE == CORRECT, we can probably continue</returns>
        private static bool CheckFileHeader(byte[] input)
        {
            bool bresult = (
                (input[00] == 0xD4) && // this is magic number 0xD4C3 B2A1
                (input[01] == 0xC3) &&
                (input[02] == 0xB2) &&
                (input[03] == 0xA1) &&

                (input[04] == 0x02) && // this is version, 0x0200 0400, actual version = 2.4
                (input[05] == 0x00) &&
                (input[06] == 0x04) &&
                (input[07] == 0x00) &&

                //(input[08] == 0x00) && // time zone
                //(input[09] == 0x00) &&
                //(input[10] == 0x00) &&
                //(input[11] == 0x00) &&

                //(input[12] == 0x00) && // accuracy of timestamps
                //(input[13] == 0x00) &&
                //(input[14] == 0x00) &&
                //(input[15] == 0x00) &&

                
                //(input[16] == 0x60) && // maximum lenngth of captured packet
                //(input[17] == 0x00) &&
                //(input[18] == 0x00) &&
                //(input[19] == 0x00) &&

                (input[20] == 0x01) && // device type (1 = ethernet)
                (input[21] == 0x00) &&
                (input[22] == 0x00) &&
                (input[23] == 0x00)
                );

            return bresult;
        }


        /// <summary>
        /// Check for unique network communications
        /// </summary>
        private void CheckForUniqueCommunications()
        {
            //
            // [1st / 3] : To Find out the Unique Communications
            //
            uniqueCommunications = new Dictionary<KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>>, object>();
            // DEF                         [0] uniqueCommunications.Key (as tree structure)           (&&)      [0] .Value (describes collection of certain packets)
            //                                          │                                                        │
            //                        ┌───────────────┴───────────────┐                                [ collection ]
            //                        │ (Key)                              │ (Value)
            //                 [1] pairs IP                           [1] pairs of ports
            //                        │                                    │
            //         ┌────────────┴──────┐                    ┌──────┴────────────┐
            //         │ (Key.Key)            │ (Key.Value)        │ (Value.Key)         │ (Value.Value)
            //   [2] local IP            [2] remote IP         [2] local port      [2] remote port




            foreach (var packet in packets)
            {   
                KeyValuePair<string, string> pairIPs = new KeyValuePair<string, string>(packet.IPLocal, packet.IPRemote); // local and remote IPs
                KeyValuePair<int, int> pairPorts = new KeyValuePair<int, int>(packet.LocalPort, packet.RemotePort);       // local and remote Ports
                KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>> combination = 
                    new KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>>(pairIPs, pairPorts);           // tree structured combination of unique IPs & Ports

                
                if (uniqueCommunications.Keys.Any(item => (item.Key.Key == packet.IPLocal) && (item.Key.Value == packet.IPRemote) && (item.Value.Key == packet.LocalPort) && (item.Value.Value == packet.RemotePort)))
                    // if "uniqueCommunications" contains an element of "combination" ...
                {
                    IList<Packet> list = (IList<Packet>)uniqueCommunications.Where(item => (item.Key.Key.Key == packet.IPLocal) && (item.Key.Key.Value == packet.IPRemote) && (item.Key.Value.Key == packet.LocalPort) && (item.Key.Value.Value == packet.RemotePort)).First().Value;
                    // ... then choose it, and add to its content this "packet"
                    list.Add(packet);
                }
                else // if doesn't ...
                {
                    IList<Packet> list = new List<Packet>();  // ... create new collection,
                    list.Add(packet);                         // add packet to this collection, ...
                    uniqueCommunications[combination] = list; // and link the "list"'s pointer to "uniqueCommunications" dictionary
                }
            }




            //
            // [2nd / 3] : To Prepare Template Collection
            //
            IDictionary<KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>>, Template> templateDictionary = new Dictionary<KeyValuePair<KeyValuePair<string, string>, KeyValuePair<int, int>>, Template>();
            foreach (var viacom in uniqueCommunications) // for every unique communication profile...
            {
                Template template = null;
                if (AppSettings.Templates.Items.Any(item => (item.LocalIP == viacom.Key.Key.Key) && (item.RemoteIP == viacom.Key.Key.Value) && (item.LocalPort == viacom.Key.Value.Key) && (item.RemotePort == viacom.Key.Value.Value))) // if this unique communication profile exists in "AppSettings.Templates" ...
                {
                    template = AppSettings.Templates.Items.Where(item => (item.LocalIP == viacom.Key.Key.Key) && (item.RemoteIP == viacom.Key.Key.Value) && (item.LocalPort == viacom.Key.Value.Key) && (item.RemotePort == viacom.Key.Value.Value)).First(); // ...then take that template from "AppSettings.Templates" and save to "template" variable,
                }
                else
                {
                    template = AppSettings.Templates.Default; // if doesn't exist, just take the defaults
                }
                templateDictionary[viacom.Key] = template; // after that, save 
            }




            //
            // [3rd / 3] : And Finally - To Get {Template} for Every Single {Packet}
            //
            templateCollection = new Dictionary<Packet, Template>();
            foreach (var com in uniqueCommunications)
            {

                IEnumerable<Packet> list = com.Value as IEnumerable<Packet>;
                if (list != null)
                {
                    foreach (var packet in list)
                    {
                        templateCollection[packet] = templateDictionary[com.Key];
                    }
                }
            }
        }

        


    }
}
