﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace PacMap.Protocols
{
    /// <summary>
    /// Protocol Names
    /// </summary>
    public enum ProtocolType
    {
        HOPOPT,
        ICMP,
        IGMP,
        GGP,
        IPv4,
        ST,
        TCP,
        CBT,
        EGP,
        IGP,
        BBN_RCC_MON,
        NVP_II,
        PUP,
        ARGUS,
        EMCON,
        XNET,
        CHAOS,
        UDP,
        MUX,
        DCN_MEAS,
        HMP,
        PRM,
        XNS_IDP,
        TRUNK_1,
        TRUNK_2,
        LEAF_1,
        LEAF_2,
        RDP,
        IRTP,
        ISO_TP4,
        NETBLT,
        MFE_NSP,
        MERIT_INP,
        DCCP,
        _3PC,
        IDPR,
        XTP,
        DDP,
        IDPR_CMTP,
        TPpp,
        IL,
        IPv6,
        SDRP,
        IPv6_Route,
        IPv6_Frag,
        IDRP,
        RSVP,
        GRE,
        DSR,
        BNA,
        ESP,
        AH,
        I_NLSP,
        SWIPE,
        NARP,
        MOBILE,
        TLSP,
        SKIP,
        IPv6_ICMP,
        IPv6_NoNxt,
        IPv6_Opts,
        CFTP,
        SAT_EXPAK,
        KRYPTOLAN,
        RVD,
        IPPC,
        SAT_MON,
        VISA,
        IPCV,
        CPNX,
        CPHB,
        WSN,
        PVP,
        BR_SAT_MON,
        SUN_ND,
        WB_MON,
        WB_EXPAK,
        ISO_IP,
        VMTP,
        SECURE_VMTP,
        VINES,
        TTP,
        NSFNET_IGP,
        DGP,
        TCF,
        EIGRP,
        OSPFIGP,
        Sprite_RPC,
        LARP,
        MTP,
        AX_25,
        IPIP,
        MICP,
        SSC_SP,
        ETHERIP,
        ENCAP,
        GMTP,
        IFMP,
        PNNI,
        PIM,
        ARIS,
        SCPS,
        QNX,
        A_N,
        IPComp,
        SNP,
        Compaq_Peer,
        IPX_IN_IP,
        VRRP,
        PGM,
        L2TP,
        DDX,
        IATP,
        STP,
        SRP,
        UTI,
        SMP,
        SM,
        PTP,
        ISISoverIPv4,
        FIRE,
        CRTP,
        CRUDP,
        SSCOPMCE,
        IPLT,
        SPS,
        PIPE,
        SCTP,
        FC,
        RSVP_E2E_IGNORE,
        MobilityHeader,
        UDPLite,
        MPLS_IN_IP,
        manet,
        HIP,
        Shim6,
        WESP,
        ROHC,
        reserved
    }


    /// <summary>
    /// Protocol Information Database
    /// </summary>
    public static class ProtocolsInfo
    {

        #region ___Dictionaries
        /// <summary>
        /// Protocol No; Name
        /// </summary>
        public static readonly IDictionary<int, ProtocolType> ProtocolsNo = new Dictionary<int, ProtocolType>()
        {
            {0, ProtocolType.HOPOPT},
            {1, ProtocolType.ICMP},
            {2, ProtocolType.IGMP},
            {3, ProtocolType.GGP},
            {4, ProtocolType.IPv4},
            {5, ProtocolType.ST},
            {6, ProtocolType.TCP},
            {7, ProtocolType.CBT},
            {8, ProtocolType.EGP},
            {9, ProtocolType.IGP},
            {10, ProtocolType.BBN_RCC_MON},
            {11, ProtocolType.NVP_II},
            {12, ProtocolType.PUP},
            {13, ProtocolType.ARGUS},
            {14, ProtocolType.EMCON},
            {15, ProtocolType.XNET},
            {16, ProtocolType.CHAOS},
            {17, ProtocolType.UDP},
            {18, ProtocolType.MUX},
            {19, ProtocolType.DCN_MEAS},
            {20, ProtocolType.HMP},
            {21, ProtocolType.PRM},
            {22, ProtocolType.XNS_IDP},
            {23, ProtocolType.TRUNK_1},
            {24, ProtocolType.TRUNK_2},
            {25, ProtocolType.LEAF_1},
            {26, ProtocolType.LEAF_2},
            {27, ProtocolType.RDP},
            {28, ProtocolType.IRTP},
            {29, ProtocolType.ISO_TP4},
            {30, ProtocolType.NETBLT},
            {31, ProtocolType.MFE_NSP},
            {32, ProtocolType.MERIT_INP},
            {33, ProtocolType.DCCP},
            {34, ProtocolType._3PC},
            {35, ProtocolType.IDPR},
            {36, ProtocolType.XTP},
            {37, ProtocolType.DDP},
            {38, ProtocolType.IDPR_CMTP},
            {39, ProtocolType.TPpp},
            {40, ProtocolType.IL},
            {41, ProtocolType.IPv6},
            {42, ProtocolType.SDRP},
            {43, ProtocolType.IPv6_Route},
            {44, ProtocolType.IPv6_Frag},
            {45, ProtocolType.IDRP},
            {46, ProtocolType.RSVP},
            {47, ProtocolType.GRE},
            {48, ProtocolType.DSR},
            {49, ProtocolType.BNA},
            {50, ProtocolType.ESP},
            {51, ProtocolType.AH},
            {52, ProtocolType.I_NLSP},
            {53, ProtocolType.SWIPE},
            {54, ProtocolType.NARP},
            {55, ProtocolType.MOBILE},
            {56, ProtocolType.TLSP},
            {57, ProtocolType.SKIP},
            {58, ProtocolType.IPv6_ICMP},
            {59, ProtocolType.IPv6_NoNxt},
            {60, ProtocolType.IPv6_Opts},
            {62, ProtocolType.CFTP},
            {64, ProtocolType.SAT_EXPAK},
            {65, ProtocolType.KRYPTOLAN},
            {66, ProtocolType.RVD},
            {67, ProtocolType.IPPC},
            {69, ProtocolType.SAT_MON},
            {70, ProtocolType.VISA},
            {71, ProtocolType.IPCV},
            {72, ProtocolType.CPNX},
            {73, ProtocolType.CPHB},
            {74, ProtocolType.WSN},
            {75, ProtocolType.PVP},
            {76, ProtocolType.BR_SAT_MON},
            {77, ProtocolType.SUN_ND},
            {78, ProtocolType.WB_MON},
            {79, ProtocolType.WB_EXPAK},
            {80, ProtocolType.ISO_IP},
            {81, ProtocolType.VMTP},
            {82, ProtocolType.SECURE_VMTP},
            {83, ProtocolType.VINES},
            {84, ProtocolType.TTP},
            {85, ProtocolType.NSFNET_IGP},
            {86, ProtocolType.DGP},
            {87, ProtocolType.TCF},
            {88, ProtocolType.EIGRP},
            {89, ProtocolType.OSPFIGP},
            {90, ProtocolType.Sprite_RPC},
            {91, ProtocolType.LARP},
            {92, ProtocolType.MTP},
            {93, ProtocolType.AX_25},
            {94, ProtocolType.IPIP},
            {95, ProtocolType.MICP},
            {96, ProtocolType.SSC_SP},
            {97, ProtocolType.ETHERIP},
            {98, ProtocolType.ENCAP},
            {100, ProtocolType.GMTP},
            {101, ProtocolType.IFMP},
            {102, ProtocolType.PNNI},
            {103, ProtocolType.PIM},
            {104, ProtocolType.ARIS},
            {105, ProtocolType.SCPS},
            {106, ProtocolType.QNX},
            {107, ProtocolType.A_N},
            {108, ProtocolType.IPComp},
            {109, ProtocolType.SNP},
            {110, ProtocolType.Compaq_Peer},
            {111, ProtocolType.IPX_IN_IP},
            {112, ProtocolType.VRRP},
            {113, ProtocolType.PGM},
            {115, ProtocolType.L2TP},
            {116, ProtocolType.DDX},
            {117, ProtocolType.IATP},
            {118, ProtocolType.STP},
            {119, ProtocolType.SRP},
            {120, ProtocolType.UTI},
            {121, ProtocolType.SMP},
            {122, ProtocolType.SM},
            {123, ProtocolType.PTP},
            {124, ProtocolType.ISISoverIPv4},
            {125, ProtocolType.FIRE},
            {126, ProtocolType.CRTP},
            {127, ProtocolType.CRUDP},
            {128, ProtocolType.SSCOPMCE},
            {129, ProtocolType.IPLT},
            {130, ProtocolType.SPS},
            {131, ProtocolType.PIPE},
            {132, ProtocolType.SCTP},
            {133, ProtocolType.FC},
            {134, ProtocolType.RSVP_E2E_IGNORE},
            {135, ProtocolType.MobilityHeader},
            {136, ProtocolType.UDPLite},
            {137, ProtocolType.MPLS_IN_IP},
            {138, ProtocolType.manet},
            {139, ProtocolType.HIP},
            {140, ProtocolType.Shim6},
            {141, ProtocolType.WESP},
            {142, ProtocolType.ROHC},
            {255, ProtocolType.reserved}
        };

        /// <summary>
        /// Protocol No; Fullnames
        /// </summary>
        public static readonly IDictionary<int, string> ProtocolsName = new Dictionary<int, string>()
        {
            {0, "IPv6 Hop-by-Hop"},
            {1, "Internet Control Message"},
            {2, "Internet Group Manager"},
            {3, "Gateway-to-Gateway"},
            {4, "IPv4 encapsulation"},
            {5, "Stream"},
            {6, "Transmission Control"},
            {7, "CBT"},
            {8, "Exterior Gateway Protocol"},
            {9, "any private interior gateway (used by Cisco for their IGRP)"},
            {10, "BBN RCC Monitoring"},
            {11, "Network Voice Protocol"},
            {12, "PUP"},
            {13, "ARGUS"},
            {14, "EMCON"},
            {15, "Cross Net Debugger"},
            {16, "Chaos"},
            {17, "User Datagram Protocol"},
            {18, "Multiplexing"},
            {19, "DCN Measurement Subsystems"},
            {20, "Host Monitoring"},
            {21, "Packet Radio Measurement"},
            {22, "XEROX NS IDP"},
            {23, "Trunk-1"},
            {24, "Trunk-2"},
            {25, "Leaf-1"},
            {26, "Leaf-2"},
            {27, "Reliable Data Protocol"},
            {28, "Internet Reliable Transaction"},
            {29, "ISO Transport Protocol Class 4"},
            {30, "Buld Data Transfer Protocol"},
            {31, "MFE Network Services Protocol"},
            {32, "MERIT Internodal Protocol"},
            {33, "Datagram Congestion Control Protocol"},
            {34, "Third Party Connect Protocol"},
            {35, "Inter-Domain Policy Routing Protocol"},
            {36, "XTP"},
            {37, "Datagram Delivery Protocol"},
            {38, "IDPR Control Message Transport Protocol"},
            {39, "TP++ Transport Protocol"},
            {40, "IL Transport Protocol"},
            {41, "IPv6 encapsulation"},
            {42, "Source Demand Routing Protocol"},
            {43, "Routing Header for IPv6"},
            {44, "Fragment Header for IPv6"},
            {45, "Inter-Domain Routing Protocol"},
            {46, "Reservation Protocol"},
            {47, "General Routing Encapsulation"},
            {48, "Dynamic Source Routing Protocol"},
            {49, "BNA"},
            {50, "Encap Securitiy Payload"},
            {51, "Authentication Header"},
            {52, "Integrated Net Layer Security TUBA"},
            {53, "IP with Encryption"},
            {54, "NBMA Address Resolution Protocol"},
            {55, "IP Mobility"},
            {56, "Transport Layer Security Protocol using Kryptonet key management"},
            {57, "SKIP"},
            {58, "ICMP for IPv6"},
            {59, "No Next Header for IPv6"},
            {60, "Destination Options for IPv6"},
            {62, "CFTP"},
            {64, "SATNET and Backroom EXPAK"},
            {65, "Kryptolan"},
            {66, "MIT Remote Virtual Disk Protocol"},
            {67, "Internet Pluribus Packet Core"},
            {69, "SATNET Monitoring"},
            {70, "VISA Protocol"},
            {71, "Internet Packet Core Utility"},
            {72, "Computer Protocol Network Executive"},
            {73, "Computer Protocol Heart Beat"},
            {74, "Wang Span Network"},
            {75, "Packet Video Protocol"},
            {76, "Backroom SATNET Monitoring"},
            {77, "SUN ND PROTOCOL Temporary"},
            {78, "WIDEBAND Monitoring"},
            {79, "WIDEBAND Expak"},
            {80, "ISO-IP"},
            {81, "VMTP"},
            {82, "SECURE-VMTP"},
            {83, "VINES"},
            {84, "TTP"},
            {85, "NSFNET-IGP"},
            {86, "Dissimilar Gateway Protocol"},
            {87, "TCF"},
            {88, "EIGRP"},
            {89, "OSPFIGP"},
            {90, "Sprite RPC Protocol"},
            {91, "Locus Address Resolution Protocol"},
            {92, "Multicast Transport Protocol"},
            {93, "AX.25 Frames"},
            {94, "IP-within-IP Encapsulation Protocol"},
            {95, "Mobile Internetworking Control Pro."},
            {96, "Semaphore Communications Sec. Pro."},
            {97, "Ethernet-within-IP Encapsulation"},
            {98, "Encapsulation Header"},
            {100, "GMTP"},
            {101, "Ipsilon Flow Management Protocol"},
            {102, "PNNI over IP"},
            {103, "Procotol Independent Multicast"},
            {104, "ARIS"},
            {105, "SCPS"},
            {106, "QNS"},
            {107, "Active Networks"},
            {108, "IP Payload Compression Protocol"},
            {109, "Sitara Networks Protocol"},
            {110, "Compaq Peer Protocol"},
            {111, "IPX in IP"},
            {112, "Virtual Router Redundancy Protocol"},
            {113, "PGM Reliable Redundancy Protocol"},
            {115, "Layer Two Tunelling Protocol"},
            {116, "D-II Data Exchange (DDX)"},
            {117, "Interactive Agent Transfer Protocol"},
            {118, "Schedule Transfer Protocol"},
            {119, "SpectraLink Radio Protocol"},
            {120, "UTI"},
            {121, "Simple Message Protocol"},
            {122, "SM"},
            {123, "Performance Transparency Protocol"},
            {124, "ISIS over IPv4"},
            {125, "FIRE"},
            {126, "Combat Radio Transport Protocol"},
            {127, "Combat Radio User Datagram"},
            {128, "SSCOPMCE"},
            {129, "IPLT"},
            {130, "Secure Packet Shield"},
            {131, "Private IP Encapsulation within IP"},
            {132, "Stream Control Transmission Protocol"},
            {133, "Fibre Channel"},
            {134, "RSVP-E2E-IGNORE"},
            {135, "Mobility Header"},
            {136, "UDP Lite"},
            {137, "MPLS-in-IP"},
            {138, "MANET Protocols"},
            {139, "Host Identity Protocol"},
            {140, "Shim6 Protocol"},
            {141, "Wrapped Encapsulating Security Payload"},
            {142, "Robust Header Compression"},
            {255, ""}
        };


                #endregion


        private static readonly IDictionary<KeyValuePair<int, ProtocolType>, string> actualPortsDB = new Dictionary<KeyValuePair<int, ProtocolType>, string>();      
        


        /// <summary>
        /// Get Protocol No
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        public static int GetProtocolNo(this ProtocolType type)
        {
            return ProtocolsNo.Where(item => item.Value == type).First().Key;
        }

        /// <summary>
        /// Get Protocol Discription Note
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        public static string GetProtocolName(this ProtocolType type)
        {
            int index = type.GetProtocolNo();
            return ProtocolsName[index];
        }


        /// <summary>
        /// Get Single Port Name or just Number Port (if not found in DB)
        /// </summary>
        /// <param name="portNo">port no</param>
        /// <param name="protocol">TCP or UDP protocol</param>
        /// <returns>single port name</returns>
        public static string GetSinglePortName(this int portNo, ProtocolType protocol)
        {
            string result = GetPortName(portNo, protocol);
            if (result == "")
                result = String.Format("{0}", portNo);
            else
            {
                result = result.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries).ToList()[0];
            }
            return result;
        }

        /// <summary>
        /// Get Ports Name
        /// </summary>
        /// <param name="portNo">port no</param>
        /// <param name="protocol">TCP or UDP protocol</param>
        /// <returns>one or group of port name(s)</returns>
        public static string GetPortName(this int portNo, ProtocolType protocol)
        {
            KeyValuePair<int, ProtocolType> lookingFor = new KeyValuePair<int, ProtocolType>(portNo, protocol);
            if (actualPortsDB.ContainsKey(lookingFor))
                return actualPortsDB[lookingFor];
            else
            {
                string actualAdded = GetPortName(portNo, protocol, true);
                actualPortsDB[lookingFor] = actualAdded;
                return actualAdded;
            }
            
        }
        
        /// <summary>
        /// Get Ports Name
        /// </summary>
        /// <param name="portNo">port no</param>
        /// <param name="protocol">TCP or UDP protocol</param>
        /// <param name="databaseListening">DB Browsing Enabled</param>
        /// <returns>one or group of port name(s)</returns>
        private static string GetPortName(this int portNo, ProtocolType protocol, bool databaseListening)
        {
            string result = "";
            Stream stream = new MemoryStream(Properties.Resources.Ports);
            stream.Position = 0;

            IList<string[]> foundedCorrectPortsLine = new List<string[]>();
            #region foundedCorrectPortsLine = Filter "Properties.Resources.Ports" for matching portNo
            {
                bool loop = true;
                string empty = "";
                bool quoteActivated = false;
                while (loop)
                {
                    int test = stream.ReadByte();
                    if (test != -1)
                    {
                        byte ch = (byte)test;

                        if ((ch == '[') && (!quoteActivated))
                        {
                            quoteActivated = true;
                        }                        
                        if ((ch != '\r') && (ch != '\n') && (!quoteActivated))
                        {
                            empty += (char)ch;
                        }
                        if ((ch == ']') && (quoteActivated))
                        {
                            quoteActivated = false;
                        }
                        if (ch == '\n')
                        {
                            bool testForSuccess = false;
                            string[] groups = empty.Split(new char[] { '\t' }, StringSplitOptions.RemoveEmptyEntries);

                            string portNoString = groups[0];
                            string[] span = portNoString.Split(new char[] { '-' }, StringSplitOptions.RemoveEmptyEntries);
                            if (span.Count() == 2) // span testing 
                            {
                                int left;
                                int right;
                                int.TryParse(span[0], out left);
                                int.TryParse(span[1], out right);
                                if ((left <= portNo) && (portNo <= right)) // must be in the span
                                    testForSuccess = true;
                            }
                            if (span.Count() == 1) // no span testing
                            {
                                int value;
                                int.TryParse(span[0], out value);
                                if (value == portNo)
                                    testForSuccess = true;
                            }


                            if (testForSuccess) // if somethings goes well -> add to the collection
                                foundedCorrectPortsLine.Add(groups);
                            empty = "";
                        }
                    }
                    else
                        loop = false;
                }
            }
            #endregion

            IList<string> filteredPortsName = new List<string>();
            #region filteredPortsName = Filter "foundedCorrectPortsLine" for given protocol
            {
                string protocolString = "!!Unspecified!!";
                if (protocol == ProtocolType.TCP)
                    protocolString = "TCP";
                if (protocol == ProtocolType.UDP)
                    protocolString = "UDP";


                IList<string[]> filteredCorrectPortsForSetProtocol = new List<string[]>();
                filteredCorrectPortsForSetProtocol = foundedCorrectPortsLine.Where(item => item.Any(i => i == protocolString)).ToList();


                foreach (var item in filteredCorrectPortsForSetProtocol)
                {
                    filteredPortsName.Add(item.Last());
                }
            }
            #endregion

            if (filteredPortsName.Count > 0)
            {
                foreach (var item in filteredPortsName)
                {
                    if (result != "")
                        result += "\r\n";
                    result += String.Format("{0}", item);
                }
            }

            return result;
        }





        /// <summary>
        /// Create Special Protocol for Inner Bytes Array Content
        /// </summary>
        /// <param name="type">protocol type</param>
        /// <param name="optionalRemotePort">optional port</param>
        /// <param name="packet">packet frame origin</param>
        /// <param name="content">byte content</param>
        /// <returns>result protocol (Ready to Operate)</returns>
        public static Protocol CreateAppropriateProtocolContent(ProtocolType type, int optionalLocalPort, int optionalRemotePort, Packet.Packet packet, byte[] content)
        {
            Protocol result = new Protocol(content, packet);

            //== BASICs ======================================================
            //
            // TCP content
            //
            if ((type == ProtocolType.TCP) && (optionalRemotePort == 0))
                result = new TCP(content, packet);

            //
            // UDP content
            //
            if ((type == ProtocolType.UDP) && (optionalRemotePort == 0))
                result = new UDP(content, packet);



            //== COMBINATIONs =================================================
            //
            // HTTP content
            //
            if ((type == ProtocolType.TCP) && (optionalRemotePort == 80))
                result = new HTTP(content, packet);


            //
            // Teredo - IPv6 over UDP Tunneling
            //
            if ((type == ProtocolType.UDP) && (optionalRemotePort == 3544)) 
                result = new IPv6(content, packet);





            return result;
        }


    }


}
