﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using PacMap.General;

namespace PacMap.General.Masks
{
    /// <summary>
    /// USER MASK Class
    /// </summary>
    public abstract class Mask
    {
        /// <summary>
        /// Construct Empty Mask
        /// </summary>
        public Mask()
        {
            isCorrect = true;
            isEmpty = true;
        }

        /// <summary>
        /// Construct User Mask
        /// </summary>
        /// <param name="byteLength">total mask length</param>
        /// <param name="maskType">mask type</param>
        /// <param name="maskedString">mask string ('true' - original, 'false' - anonmapping)</param>
        public Mask(int byteLength, MaskEnum maskType, string maskedString)
        {
            length = byteLength;
            type = maskType;

            int expectedStringLength = 0;
            if (type == MaskEnum.Hex)
                expectedStringLength = length * 2;
            if (type == MaskEnum.Bin)
                expectedStringLength = length * 8;
            if (maskedString.Length != expectedStringLength) // bad input happened
            {
                lastError = "Bad Mask Length";
                return;
            }

            if (type == MaskEnum.Hex)
            {
                mask = maskedString.ConvertHexStringToBytesArray();
            }
            if (type == MaskEnum.Bin)
            {
                if (maskedString.Any(item => ((item != '1') && (item != '0'))))
                    return;

                IList<byte> maskList = new List<byte>();
                int count = maskedString.Length / 8;
                for (int i = 1; i <= count; i++)
                {
                    byte innerResult = 0x00;
                    int j = (i - 1) * 8;
                    int temporal = 0;

                    for (int k = 1; k <= 8; k++)
                    {
                        int.TryParse(maskedString[j + k - 1].ToString(), out temporal);
                        innerResult = (byte)(temporal | (temporal << (8-k)));
                    }

                    maskList.Add(innerResult);
                }

                mask = maskList.ToArray();
            }


            isCorrect = true;
        }
        


        /**********************************************************************************************/
        // :: DATA FIELDs ::
        protected bool isCorrect = false;
        protected bool isEmpty = false;
        protected int length = 0;
        protected byte[] mask = null;
        private MaskEnum type = MaskEnum.Hex;
        protected string lastError = null;
        public enum MaskEnum
        {
            Hex,
            Bin
        }

        /**********************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Mask Is Empty
        /// </summary>
        public bool IsEmpty
        {
            get
            {
                return isEmpty;
            }
        }

        /// <summary>
        /// Mask Is Correct
        /// </summary>
        public bool IsCorrect
        {
            get
            {
                return isCorrect;
            }
        }

        /// <summary>
        /// Get Mask Length
        /// </summary>
        public int Length
        {
            get
            {
                return mask.Length;
            }
        }

        /// <summary>
        /// Get Last Error
        /// </summary>
        public string LastError
        {
            get
            {
                if (lastError == null)
                    return "";
                else
                    return lastError;
            }
        }
        #endregion

        /**********************************************************************************************/
        // :: GLOBAL METHODs ::
        /// <summary>
        /// Proceed Mask on Input Item
        /// </summary>
        /// <param name="original">original data</param>
        /// <param name="anonmapped">anonmapped data</param>
        /// <returns>result string</returns>
        public virtual byte[] Proceed(byte[] original, byte[] anonmapped)
        {
            if ((IsEmpty) || (original.Length != anonmapped.Length) || (original.Length != mask.Length))
                return original;

            byte[] result = new byte[mask.Length];

            byte[] inverseMask = new byte[mask.Length];
            for (int i = 1; i <= mask.Length; i++) // preparing inverse mask
            {
                inverseMask[i - 1] = (byte)((~mask[i - 1]) & 0xFF);
            }



            for (int i = 1; i <= mask.Length; i++) // proceeding mask
            {
                result[i-1]  = (byte)((mask[i-1] & original[i-1]) | (inverseMask[i-1] & anonmapped[i-1]));
            }

            return result;
        }


        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>string result</returns>
        public override string ToString()
        {
            if (!IsCorrect)
                return "badmask";
            if (IsEmpty)
                return "none";

            string result = "";
            if (type == MaskEnum.Hex)
            {
                result += "0x";
                foreach (var item in mask)
                {
                    string temp = String.Format("{0:X}", item);
                    if (temp.Length == 1)
                        temp = "0" + temp;

                    result += temp;
                }
            }

            if (type == MaskEnum.Bin)
            {
                foreach (var item in mask)
                {
                    string innerResult = "";
                    IList<int> temp = new List<int>(){0,0,0,0,0,0,0,0};

                    int max = 0x80;
                    for (int i = 1; i <= 8; i++)
                    {
                        if (((item & (max/i)) >> (8-i)) == 0x01)
                            temp[i-1] = 1;

                        innerResult += temp[i - 1].ToString();
                    }

                    result += innerResult;
                }
                result += "b";
            }

            return result;
        }


        /// <summary>
        /// Parse Input String to Get Mask Length, Mask Type and Mask String
        /// </summary>
        /// <param name="input">inout string</param>
        /// <param name="lng">mask length</param>
        /// <param name="mskType">mask type</param>
        /// <param name="mskString">mask string</param>
        /// <returns>TRUE = if success</returns>
        protected static bool ParseBinHex(string input, out int lng, out MaskEnum mskType, out string mskString)
        {
            MaskEnum type = MaskEnum.Hex;
            string maskedString = "";
            int byteLength = 0;
            bool isCorrect = false;

            if (input.Length > 2)
            {
                string testForHex = input.Substring(0, 2);
                if (testForHex == "0x")
                {
                    isCorrect = true;
                    type = MaskEnum.Hex;
                    maskedString = input.Substring(2, input.Length - 2);
                    byteLength = maskedString.Length / 2;
                }
            }
            if (input.Length > 1)
            {
                string testForBin = input.Substring(input.Length - 1, 1);
                if (testForBin == "b")
                {
                    isCorrect = true;
                    type = MaskEnum.Bin;
                    maskedString = input.Substring(0, input.Length - 1);
                    byteLength = maskedString.Length / 8;
                }
            }

            if (isCorrect)
            {
                lng = byteLength;
                mskType = type;
                mskString = maskedString;
            }
            else
            {
                lng = 0;
                mskType = MaskEnum.Hex;
                mskString = "";
            }

            return isCorrect;
        }

        /// <summary>
        /// Create New Mask from String
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>completed mask</returns>
        //public static Mask Parse(string input)
        //{
        //    Mask result = null;
        //    int byteLength;
        //    MaskEnum type;
        //    string maskedString;
        //    bool isCorrect = ParseBinHex(input, out byteLength, out type, out maskedString);

        //    if (!isCorrect)
        //        result = new Mask();
        //    else
        //        result = new Mask(byteLength, type, maskedString);
        //    return result;
        //}


      


    }
}
