﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;


namespace PacMap.General.Masks
{
    /// <summary>
    /// USER IPv4 Address MASK Class
    /// </summary>
    public class MaskURI : Mask
    {
        /// <summary>
        /// Construct Empty URI Mask
        /// </summary>
        public MaskURI()
            : base()
        {

        }

        /// <summary>
        /// Construct User URI Mask
        /// </summary>
        /// <param name="byteLength">total mask length</param>
        /// <param name="maskType">mask type</param>
        /// <param name="maskedString">mask string ('true' - original, 'false' - anonmapping)</param>
        public MaskURI(int byteLength, MaskEnum maskType, string maskedString)
            : base(byteLength, maskType, maskedString)
        {
        }


        /// <summary>
        /// Construct User URI Mask from simple texted string ((mandatory form: "-.X.X") ('X' - original, '-' anonmapping) or 3 or "X.-.-" or 3!)
        /// </summary>
        /// <param name="input">mask string</param>
        public MaskURI(string input)
        {
            if ((input == "none") || (input == "0"))
            {
                isEmpty = true;
                isCorrect = true;
                return;
            }


            IList<bool> masking = null;
            isEmpty = false;
            isCorrect = false;
            isSpecialized = true;

            #region Abbreviation expression
            //
            // Test for abbreviation expression
            //
            if (input.Length >= 1)
            {
                bool leftExl = false;
                bool rightExl = false;
                bool hasNumber = false;
                if ((input.Substring(0, 1) == "!") && (input.Length >= 2))
                {
                    leftExl = true; // left exclude/include?
                    input = input.Substring(1, input.Length - 1);
                }
                if ((input.Substring(input.Length - 1, 1) == "!") && (input.Length >= 2))
                {
                    rightExl = true; // right exclude/include?
                    input = input.Substring(0, input.Length - 1);
                }
                int num;
                hasNumber = int.TryParse(input, out num);
                if (hasNumber)
                {
                    if (num < 1)
                        hasNumber = false;
                }



                //
                // Handling it
                //
                if (hasNumber)
                {
                    masking = new List<bool>();
                    for (int i = 1; i <= num + 1; i++)
                    {
                        bool acceptance = false;
                        if (i < num)
                        {
                            if (rightExl)
                                acceptance = true;
                        }
                        else if (i == num)
                        {
                            acceptance = true;
                        }
                        else if (i > num)
                        {
                            if (leftExl)
                                acceptance = true;
                        }

                        masking.Add(acceptance);
                        isCorrect = true;
                    }


                }
            }
            #endregion

            #region String expression
            if ((input.Length >= 1) && (!isCorrect))
            {
                string[] groups = input.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries);
                if (groups.All(item => (item == "X") || (item == "-")))
                {
                    masking = new List<bool>();
                    for (int i = 1; i <= groups.Count(); i++)
                    {
                        bool acceptance = false;
                        if (groups[i - 1] == "X")
                            acceptance = true;
                        masking.Add(acceptance);
                    }
                    isCorrect = true;
                }
            }
            #endregion



            if (isCorrect)
            {
                IList<byte> newMask = new List<byte>();
                foreach (var item in masking)
                {
                    byte test = 0x00;
                    if (item)
                        test = 0xFF;
                    newMask.Add(test);
                }
                mask = newMask.ToArray();
                length = mask.Length;
            }





        }

        

        /**********************************************************************************************/
        // :: DATA FIELDs ::
        private bool isSpecialized = false;

        /**********************************************************************************************/
        // :: PROPERTIEs ::


        /**********************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Proceed Mask on Input Item
        /// </summary>
        /// <param name="original">original data</param>
        /// <param name="anonmapped">anonmapped data</param>
        /// <returns>result string</returns>
        public override byte[] Proceed(byte[] original, byte[] anonmapped)
        {
            if ((IsEmpty))// || (original.Length != anonmapped.Length) || (original.Length != mask.Length))
                return anonmapped;

            if (!isSpecialized)
                return anonmapped;
            else
            {
                byte[] result = anonmapped;
           
                string originalString = original.ConvertToString();
                string anonmappedString = anonmapped.ConvertToString();

                string[] originalDomain = originalString.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries).Reverse().ToArray();
                string[] anonmappedDomain = anonmappedString.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries).Reverse().ToArray();

                if (originalDomain.Count() == anonmappedDomain.Count())
                {
                    IList<string> resultDomain = new List<string>();
                    for (int i = 1; i <= anonmappedDomain.Length; i++ )
                    {
                        bool isAnonMapping = true;
                        if (i <= length)
                        {
                            if (mask[i - 1] == 0xFF)
                                isAnonMapping = false;
                        }
                        else
                        {
                            if (mask.Last() == 0xFF)
                                isAnonMapping = false;
                        }




                        if (isAnonMapping)
                            resultDomain.Add(anonmappedDomain[i-1]);
                        else
                            resultDomain.Add(originalDomain[i-1]);
                    }

                    string resultString = String.Join(".", resultDomain.Reverse());
                    result = resultString.ConvertStringToBytes().ToArray();
                }
                


                return result;
            }
        }

        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>string result</returns>
        public override string ToString()
        {
            if (!IsCorrect)
                return "badmask";
            if (IsEmpty)
                return "none";

            string result = "";
            if (!isSpecialized)
                result = base.ToString();
            else
            {
                IList<string> groups = new List<string>();
                for (int i = 1; i <= mask.Length; i++)
                {
                    string item = "-";
                    if (mask[i - 1] == 0xFF)
                        item = "X";

                    groups.Add(item);
                }

                groups = groups.Reverse().ToList();
                result = string.Join(".", groups);

            }

            return result;
        }

        /// <summary>
        /// Create New Mask from String
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>completed mask</returns>
        public static MaskURI Parse(string input)
        {
            MaskURI result = null;
            bool anyError = true;

            int length;
            MaskEnum type;
            string maskedString;
            anyError = !Mask.ParseBinHex(input, out length, out type, out maskedString);
            if (anyError)
            {
                result = new MaskURI(input);
            }
            else
            {
                result = new MaskURI(length, type, maskedString);
            }
            return result;
        }



    }
}
