﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using PacMap.General;
using PacMap.Protocols;
using PacMap.Anonymize;
using System.Drawing;
using PacMap.Anonymize.Templates;

namespace PacMap.Packet
{
    /// <summary>
    /// Single Packet Class
    /// </summary>
    public class Packet
    {

        /// <summary>
        /// Private Initialization of Packet object
        /// </summary>
        private Packet()
        {
            MACSource = "";
            MACDestination = "";
            Protocol = "";
            IPSource = "";
            IPDestination = "";
            LocalPort = 0;
            RemotePort = 0;
            Info = "";
            ProtocolNo = 0;
        }
        
        /// <summary>
        /// Create new Packet class
        /// </summary>
        /// <param name="stream">stream of bytes of single packet to make parsing</param>
        public Packet(Stream stream) : this()
        {
            totalLength = (int)stream.Length;

            byte[] packetInfoBuffer = new byte[16]; // has 16 bytes of packet frame
            stream.Read(packetInfoBuffer, 0, 16);
            frame = new Frame(packetInfoBuffer); // get captured epoch time; original and captured total length of actual packet

            //
            // ??? Get Transmission Direction (contains source and destination MAC ADDRESSes)
            //
            {
                stream.Read(direction, 0, 12);
                // 12 bytes contains: (Legend: '?' means any byte value)
                if (direction[6] == 'L')                  // ]012345678901[
                {                                         // ]REMOTELOCAL [ ideal example
                    Direction = Direction.RemoteToLocal;  // ]??????L?????[ general example
                }
                else if (direction[0] == 'L')             // ]012345678901[
                {                                         // ]LOCAL REMOTE[ ideal example
                    Direction = Direction.LocalToRemote;  // ]L???????????[ general example
                }
                else
                {
                    lastError = String.Format("Unknown (LOCAL REMOTE) or (REMOTELOCAL ) at position {0}", stream.Position - 12);
                    return;
                }
            }

            //
            // Get group and individual IGs & LGs
            //
            {
                IGd = (((int)direction[1] & 0x01) == 0x01) ? true : false;
                LGd = (((int)direction[1] & 0x02) >> 1 == 0x01) ? true : false;
                IGs = (((int)direction[7] & 0x01) == 0x01) ? true : false;
                LGs = (((int)direction[7] & 0x02) >> 1 == 0x01) ? true : false;
                for (int i = 0; i <= 5; i++)
                {
                    string text1 = String.Format("{0:X}", direction[i]);
                    string text2 = String.Format("{0:X}", direction[i+6]);
                    if (text1.Length == 1)
                        text1 = "0" + text1;
                    if (text2.Length == 1)
                        text2 = "0" + text2;

                    if (MACSource != "")
                        MACSource += ":";
                    MACSource += text1;

                    if (MACDestination != "")
                        MACDestination += ":";
                    MACDestination += text2;
                }
            }

            //
            // Test for - EthernetII - WIRE
            //
            {
                byte[] bytes = new byte[2];
                stream.Read(bytes, 0, 2);
                isEth2 = (
                    (bytes[0] == 0x08) &&
                    (bytes[1] == 0x00));
                if (!isEth2)
                {
                    lastError = String.Format("Aplication understands in this time only EthernetII network wire standards at position {0}", stream.Position - 2);
                    return;
                }
            }


            //
            // Export Rest of Bytes to some Inner Protocol
            //
            {
                int word = stream.ReadByte();
                int disicion = (word & (0xF0)) >> 4;
                if (disicion == 4) // Decide to create new IPv4
                {
                    stream.Position--;
                    int restSize = (int)(stream.Length - stream.Position); // count rest size to read
                    byte[] restBuffer = new byte[restSize];                // prepare buffer
                    stream.Read(restBuffer, 0, restSize);                  // read then
                    treeroot = new IPv4(restBuffer, this);                 // construct IPv4 class
                }
                else if (disicion == 6) // Decide to create new IPv6
                {
                    stream.Position--;
                    int restSize = (int)(stream.Length - stream.Position); // count rest size to read
                    byte[] restBuffer = new byte[restSize];                // prepare buffer
                    stream.Read(restBuffer, 0, restSize);                  // read then
                    treeroot = new IPv6(restBuffer, this);                 // construct IPv4 class
                }
            }
        }

        /******************************************************************************************/
        // :: DATA FIELDs ::
        private Template template = null;
        private Frame frame = null;
        private bool isEth2 = false; // SUBLAYER - LINK LAYER - MAC (Ethernet II)
        private bool IGs = false; // FALSE = Individual Address || TRUE = Group Address
        private bool LGs = false; // FALSE = GLOBALLY ADMINISTRED ADDRESS || TRUE = GLOBALLY ADMINISTRED ADDRESS
        private bool IGd = false;
        private bool LGd = false;
        private int totalLength = 0;
        private byte[] direction = new byte[12];        
        private Protocol treeroot = null; 
        
        private string lastError = "";

        /******************************************************************************************/
        // :: PROPERTIEs ::
        public Direction Direction { get; private set; }
        /// <summary>
        /// Get Frame distribution
        /// </summary>
        public Frame Frame
        {
            get
            {
                return frame;
            }
        }

        /// <summary>
        /// Get Anonmapping Settings for This Packet
        /// </summary>
        public Template AnonSettings
        {
            get
            {
                return template;
            }
        }

        public string IPLocal
        {
            get
            {
                if (Direction == PacMap.Packet.Direction.LocalToRemote)
                    return IPSource;
                else
                    return IPDestination;
            }
        }

        public string IPRemote
        {
            get
            {
                if (Direction == PacMap.Packet.Direction.RemoteToLocal)
                    return IPSource;
                else
                    return IPDestination;
            }
        }

        public int ProtocolNo { get; set; }
        public string Protocol { get; set; }
        public string MACSource { get; set; }
        public string MACDestination { get; set; }
        public string IPSource { get; set; }
        public string IPDestination { get; set; }
        public int    LocalPort { get; set; }
        public int    RemotePort { get; set; }
        public string PortInfo { get; set; }
        public string Info { get; set; }
        

        /******************************************************************************************/
        // :: FUNCTIONs ::

        /// <summary>
        /// Make AnonMappings on actual packet
        /// </summary>
        public void Anonymize(Template anonmappingSettings)
        {
            template = anonmappingSettings;

            if (treeroot != null)
                treeroot.Anonymize();



        }


        /// <summary>
        /// Export Packets into the Bytes Array
        /// </summary>
        /// <returns>array of bytes</returns>
        public byte[] Save()
        {
            byte[] export = null;
            if (treeroot != null)
            {
                export = treeroot.Save();
            }

            // --------------------
            byte[] result = null;
            if ((treeroot != null) && (export != null))
            {
                totalLength = 16 + 12 + 2 + export.Length; // 16 frame length + 12 local->remote length + 2 Ethernet II type length
                frame.ChangeFrameLenghtInformation(export.Length + 12 + 2, export.Length + 12 + 2);
            }

            result = new byte[totalLength];
            Stream stream = new MemoryStream(result);

            stream.Position = 0;
            stream.Write(frame.Save(), 0, 16);
            stream.Write(direction, 0, 12); // local remote

            {
                int word = 0;
                if (isEth2)
                    word = 0x0800;
                else
                    word = 0;

                stream.WriteByte((byte)((word & 0xFF00) >> 8));
                stream.WriteByte((byte)(word & 0x00FF));
            }


            // ------------------------------
            if ((treeroot != null) && (export != null))
            {   
                stream.Write(export, 0, export.Length);
            }

            return result;
        }


        /// <summary>
        /// Get Full Packet Information for Advanced Packet Viewing
        /// </summary>
        /// <returns>Specialized Tree Information Object</returns>
        public object GetFullInformation()
        {
            IList<object> result = new List<object>();
            IDictionary<string, object> browserInfo = new Dictionary<string, object>();
            IList<KeyValuePair<Point, object>> browserHex = new List<KeyValuePair<Point, object>>();
            IList<string> browserGroup = new List<string>();
            result.Add(browserInfo);
            result.Add(browserHex);
            result.Add(browserGroup);
            result.Add(Save());
            int actualSeedPosition = 0;

            #region __GET FRAME INFORMATION
            //
            // Get Frame Information
            //
            {
                string name = String.Format("Frame: {0} on wire ({1} bits), {2} captured ({3} bits)", ((long)frame.OriginalPacketSize).ByteSize(), frame.OriginalPacketSize * 8, ((long)frame.CapturedPacketSize).ByteSize(), frame.CapturedPacketSize * 8);
                string item1 = String.Format("Frame Length: {0} ({1} bits)", ((long)frame.OriginalPacketSize).ByteSize(), frame.OriginalPacketSize * 8);
                string item2 = String.Format("Captured Length: {0} ({1} bits)", ((long)frame.CapturedPacketSize).ByteSize(), frame.CapturedPacketSize * 8);
                IDictionary<string, object> inner = new Dictionary<string, object>();
                inner[item1] = null;
                inner[item2] = null;
                browserInfo[name] = inner;

                Point nameHex = new Point(1, 16);
                Point item1Hex = new Point(9, 10);
                Point item2Hex = new Point(13, 14);
                IList<KeyValuePair<Point, object>> innerHex = new List<KeyValuePair<Point, object>>();
                innerHex.Add(new KeyValuePair<Point, object>(item1Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item2Hex, null));
                browserHex.Add(new KeyValuePair<Point, object>(nameHex, innerHex));

                browserGroup.Add("Frame");

                actualSeedPosition += 16;
            }
            #endregion

            #region __GET ETHERNETII INFORMATION
            //
            // Get Ethernet
            //
            {
                string macType = "";
                if (isEth2)
                    macType = "Ethernet II, ";
                string name = String.Format("{0}Src: {1}, Dst: {2}", macType, MACSource.ToLower(), MACDestination.ToLower());
                string item1 = String.Format("Destination: {0}", MACDestination);
                string item2 = String.Format("Source: {0}", MACSource);
                string item3Content = "";
                if (isEth2)
                    item3Content = "IP (0x0800)";
                string item3 = "Type: " + item3Content;
                IDictionary<string, object> inner = new Dictionary<string, object>();
                inner[item1] = null;
                inner[item2] = null;
                inner[item3] = null;
                browserInfo[name] = inner;

                Point nameHex = new Point(actualSeedPosition + 1, actualSeedPosition + 14);
                Point item1Hex = new Point(actualSeedPosition + 1, actualSeedPosition + 6);
                Point item2Hex = new Point(actualSeedPosition + 7, actualSeedPosition + 12);
                Point item3Hex = new Point(actualSeedPosition + 13, actualSeedPosition + 14);
                IList<KeyValuePair<Point, object>> innerHex = new List<KeyValuePair<Point, object>>();
                innerHex.Add(new KeyValuePair<Point, object>(item1Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item2Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item3Hex, null));
                browserHex.Add(new KeyValuePair<Point, object>(nameHex, innerHex));

                browserGroup.Add("Ethernet");

                actualSeedPosition += 14;
            }
            #endregion

            #region __GET ENCAPSULATED PROTOCOLS INFORMATION
            //
            // Get Inner Objects
            //
            {
                IList<Protocol> list = new List<Protocol>();
                if (treeroot != null)
                {
                    Protocol child = treeroot;
                    list.Add(child);


                    while (child.HasChild)
                    {
                        child = child.Child;
                        list.Add(child);
                    }

                }

                foreach (var protocol in list)
                {
                    object packetInfo = protocol.GetFullInformation();

                    if (packetInfo != null)
                    {
                        IList<object> infos = (IList<object>)packetInfo;
                        IDictionary<string, object> info = (IDictionary<string, object>)infos[0];
                        IList<KeyValuePair<Point, object>> hex = (IList<KeyValuePair<Point, object>>)infos[1];
                        IList<string> group = (IList<string>)infos[2];
                        int movingseedRightLength = hex[0].Key.Y;

                        hex = hex.MoveHexPositions(actualSeedPosition);

                        browserInfo[info.Keys.ToList()[0]] = info.Values.ToList()[0];
                        browserHex.Add(hex[0]);
                        browserGroup.Add(group[0]);

                        actualSeedPosition += movingseedRightLength;
                    }
                }
            }
            #endregion

            return result;
        }


        /// <summary>
        /// To String
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            string result = "Ethernet II Frame";
            if (treeroot != null)
            {
                result = String.Format("{0} -> {1}  {2}", IPSource, IPDestination, Protocol);
            }
            return result;
        }
    }
}
