﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using PacMap.Anonymize;
using PacMap.General;
using System.Drawing;

namespace PacMap.Protocols
{
    public class TCP : Protocol
    {
        /// <summary>
        /// Constructs TCP Class
        /// </summary>
        /// <param name="input">bytes array</param>
        /// <param name="packet">packet frame</param>
        public TCP(byte[] input, Packet.Packet packet) : base(input, packet)
        {
            shortName = "TCP";
            Stream stream = new MemoryStream(input);

            //
            // Get Source Port
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                SourcePort = (word[0] << 8) | word[1];
            }

            //
            // Get Destination Port
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                DestinationPort = (word[0] << 8) | word[1];
            }


            //
            // Get Sequence Number
            //
            {
                byte[] dword = new byte[4];
                stream.Read(dword, 0, 4);
                sequenceN = (uint)((dword[0] << 24) | (dword[1] << 16) | (dword[2] << 8) | dword[3]);
            }


            //
            // Get acknowledgment Number
            //
            {
                byte[] dword = new byte[4];
                stream.Read(dword, 0, 4);
                acknowledgmentN = (uint)((dword[0] << 24) | (dword[1] << 16) | (dword[2] << 8) | dword[3]);
            }


            //
            // Get Data Offset & 3x reserved & __cb_ns
            //
            {
                int halfbyte = stream.ReadByte();
                int leftbyte = (halfbyte & 0xF0) >> 4;
                dataOffset = leftbyte;

                int middlebyte = (halfbyte & 0x0E) >> 3;
                if (middlebyte != 0)
                {
                    return;
                }
                if ((halfbyte & 0x01) == 1)
                    __cb_ns = true;
            }

            //
            // Get Other Flags
            //
            {
                int onebyte = stream.ReadByte();
                if (((0x80 & onebyte) >> 7) == 1)
                    __cb_cwr = true;
                if (((0x40 & onebyte) >> 6) == 1)
                    __cb_ece = true;
                if (((0x20 & onebyte) >> 5) == 1)
                    __cb_urg = true;
                if (((0x10 & onebyte) >> 4) == 1)
                    __cb_ack = true;
                if (((0x08 & onebyte) >> 3) == 1)
                    __cb_psh = true;
                if (((0x04 & onebyte) >> 2) == 1)
                    __cb_rst = true;
                if (((0x02 & onebyte) >> 1) == 1)
                    __cb_syn = true;
                if ((0x01 & onebyte) == 1)
                    __cb_fin = true;
            }

            //
            // Get Window Size
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                windowSize = (word[0] << 8) | word[1];
            }

            //
            // Get Checksum
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                checksum = (word[0] << 8) | word[1];
            }

            //
            // Get Urgent Pointer
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                urgentPointer = (word[0] << 8) | word[1];
            }

            //
            // Get Additional Options
            //
            {
                if (IHL > 20)
                {
                    int additionalSize = IHL - 20;
                    hOptions = new byte[additionalSize];
                    stream.Read(hOptions, 0, additionalSize);
                }
            }


            //
            // Header Checksum 
            //


            //
            // Additionals Information exporting to packet
            //
            packet.LocalPort = Port;
            packet.RemotePort = RemotePort;
            {
                #region Making Protocol Summary
                string flagsString = GetFlagsInformation();

                string info2 = String.Format("Seq={0}", sequenceN);
                if (__cb_ack)
                    info2 += String.Format(" Ack={0}", acknowledgmentN);
                info2 += String.Format(" Win={0}", windowSize);
                #endregion
                mainInformation = String.Format("$07{0} > {1}$08{2} $07{3}", SourcePort.GetSinglePortName(ProtocolType.TCP), DestinationPort.GetSinglePortName(ProtocolType.TCP), flagsString, info2);
                packet.Info = mainInformation;
                
            }
            //--------------------------------------------
            int restSize = (int)(stream.Length - stream.Position);
            byte[] rest = new byte[restSize];
            int readed = stream.Read(rest, 0, restSize);

            if (readed != 0)
                child = ProtocolsInfo.CreateAppropriateProtocolContent(ProtocolType.TCP, Port, RemotePort, packet, rest);

            if (packet.Info == mainInformation)
            {
                packet.Protocol = "$02" + packet.Protocol;
            }
        }


        /******************************************************************************************/
        // :: DATA FIELDs ::
        //
        //=== TCP HEADER
        private int localPort = 0x0000;
        private int remotePort = 0x0000;
        private uint sequenceN = 0x00000000;
        private uint acknowledgmentN = 0x00000000;
        private int dataOffset = 0x0;
        //<== Control Bits (also known as flags)
        private bool __cb_ns  = false; //ECN-nonce concealment protection
        private bool __cb_cwr = false; //Congestion Window Reduced
        private bool __cb_ece = false; //ECN-Echo indicates
        private bool __cb_urg = false; //indicates that the Urgent pointer field is significant
        private bool __cb_ack = false; //indicates that the Acknowledgment field is significant. All packets after the initial SYN packet sent by the client should have this flag set.
        private bool __cb_psh = false; //Push function. Asks to push the buffered data to the receiving application.
        private bool __cb_rst = false; //Reset the connection
        private bool __cb_syn = false; //Synchronize sequence numbers. Only the first packet sent from each end should have this flag set. Some other flags change meaning based on this flag, and some are only valid for when it is set, and others when it is clear.
        private bool __cb_fin = false; //No more data from sender
        //> Control Bits -->
        private int windowSize = 0x0000; //the size of the receive window
        private int checksum = 0x0000; //The 16-bit checksum field is used for error-checking of the header and data
        private int urgentPointer = 0x0000; //if the URG flag is set, then this 16-bit field is an offset from the sequence number indicating the last urgent data byte
        private byte[] hOptions = null;
        //----------------------------------
        //private string protocol = "";

        /******************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Get or Set Port
        /// </summary>
        public int Port
        {
            get
            {
                return localPort;
            }
            set
            {
                localPort = value;
            }
        }

        /// <summary>
        /// Get or Set Remote Port
        /// </summary>
        public int RemotePort
        {
            get
            {
                return remotePort;
            }
            set
            {
                remotePort = value;
            }
        }

        /// <summary>
        /// Get or Set Source Port
        /// </summary>
        public int SourcePort
        {
            get
            {
                if (packet.Direction == Packet.Direction.LocalToRemote)
                    return localPort;
                else
                    return remotePort;
            }
            set
            {
                if (packet.Direction == Packet.Direction.LocalToRemote)
                    localPort = value;
                else
                    remotePort = value;
            }
        }

        /// <summary>
        /// Get or Set Destination Port
        /// </summary>
        public int DestinationPort
        {
            get
            {
                if (packet.Direction == Packet.Direction.RemoteToLocal)
                    return localPort;
                else
                    return remotePort;
            }
            set
            {
                if (packet.Direction == Packet.Direction.RemoteToLocal)
                    localPort = value;
                else
                    remotePort = value;
            }
        }

        /// <summary>
        /// Get Internal Header Length
        /// </summary>
        public int IHL
        {
            get
            {
                return dataOffset * 4;
            }
        }


        /// <summary>
        /// Get Flags Bytes Mask = 000X XXXX XXXX
        /// </summary>
        public int Flags
        {
            get
            {
                int result = 0;

                result += ((__cb_ns) ? (0x01) : (0x00)) << 8;
                result += ((__cb_cwr) ? (0x01) : (0x00)) << 7;
                result += ((__cb_ece) ? (0x01) : (0x00)) << 6;
                result += ((__cb_urg) ? (0x01) : (0x00)) << 5;
                result += ((__cb_ack) ? (0x01) : (0x00)) << 4;
                result += ((__cb_psh) ? (0x01) : (0x00)) << 3;
                result += ((__cb_rst) ? (0x01) : (0x00)) << 2;
                result += ((__cb_syn) ? (0x01) : (0x00)) << 1;
                result += ((__cb_fin) ? (0x01) : (0x00));

                return result;
            }
        }
        #endregion


        /******************************************************************************************/
        // :: PRIVATE FUNCTIONs ::

        /// <summary>
        /// Get Actual IPv4 Header in Bytes
        /// </summary>
        /// <returns>Array of bytes (20 - 60B)</returns>
        private byte[] CreateHeader()
        {
            byte[] result = new byte[IHL];
            Stream stream = new MemoryStream(result);
            stream.Position = 0;

            {
                byte[] word = new byte[2];
                word[0] = (byte)((SourcePort & 0xFF00) >> 8);
                word[1] = (byte)(SourcePort & 0x00FF);
                stream.Write(word, 0, 2);
            }

            {
                byte[] word = new byte[2];
                word[0] = (byte)((DestinationPort & 0xFF00) >> 8);
                word[1] = (byte)(DestinationPort & 0x00FF);
                stream.Write(word, 0, 2);
            }

            {
                byte[] dword = new byte[4];
                dword[0] = (byte)((sequenceN & 0xFF000000) >> 24);
                dword[1] = (byte)((sequenceN & 0x00FF0000) >> 16);
                dword[2] = (byte)((sequenceN & 0x0000FF00) >> 8);
                dword[3] = (byte)(sequenceN & 0x000000FF);
                stream.Write(dword, 0, 4);
            }

            {
                byte[] dword = new byte[4];
                dword[0] = (byte)((acknowledgmentN & 0xFF000000) >> 24);
                dword[1] = (byte)((acknowledgmentN & 0x00FF0000) >> 16);
                dword[2] = (byte)((acknowledgmentN & 0x0000FF00) >> 8);
                dword[3] = (byte)(acknowledgmentN & 0x000000FF);
                stream.Write(dword, 0, 4);
            }

            {
                byte flag = (byte)(dataOffset << 4);
                byte temp = 0;
                if (__cb_ns)
                    temp = 1;
                flag = (byte)(flag | temp);
                stream.WriteByte(flag);
            }

            {
                byte flags = 0;
                flags = (byte)(Flags & 0x00FF);
                stream.WriteByte(flags);
            }

            {
                byte[] word = new byte[2];
                word[0] = (byte)((windowSize & 0xFF00) >> 8);
                word[1] = (byte)(windowSize & 0x00FF);
                stream.Write(word, 0, 2);
            }

            {
                byte[] word = new byte[2];
                word[0] = (byte)((checksum & 0xFF00) >> 8);
                word[1] = (byte)(checksum & 0x00FF);
                stream.Write(word, 0, 2);
            }

            {
                byte[] word = new byte[2];
                word[0] = (byte)((urgentPointer & 0xFF00) >> 8);
                word[1] = (byte)(urgentPointer & 0x00FF);
                stream.Write(word, 0, 2);
            }

            if (IHL > 20)
            {
                stream.Write(hOptions, 0, hOptions.Length);
            }

            return result;
        }

        /// <summary>
        /// Get Flags Information
        /// </summary>
        /// <returns>string of activated flags</returns>
        private string GetFlagsInformation()
        {
            string result = "";
            IList<string> flagsPreparing = new List<string>();
            if (__cb_ns)
                flagsPreparing.Add("NS");
            if (__cb_cwr)
                flagsPreparing.Add("CWR");
            if (__cb_ece)
                flagsPreparing.Add("ECE");
            if (__cb_urg)
                flagsPreparing.Add("URG");
            if (__cb_ack)
                flagsPreparing.Add("ACK");
            if (__cb_psh)
                flagsPreparing.Add("PSH");
            if (__cb_rst)
                flagsPreparing.Add("RST");
            if (__cb_syn)
                flagsPreparing.Add("SYN");
            if (__cb_fin)
                flagsPreparing.Add("FIN");
            if (flagsPreparing.Count != 0)
            {
                foreach (var flag in flagsPreparing)
                {
                    if (result != "")
                        result += ", ";
                    result += flag;
                }
                result = " [" + result + "]";
            }

            return result;
        }





        /******************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Save Protocol to bytes array
        /// </summary>
        /// <returns>bytes array</returns>
        public override byte[] Save()
        {
            byte[] childOutput = null;

            if (HasChild)
            {
                childOutput = child.Save();
            }

            //-------------------------------------


            byte[] result = null;
            if ((HasChild) && (childOutput != null))
            {
                result = new byte[IHL + childOutput.Length];
            }
            else
                result = new byte[content.Length];

            Stream stream = new MemoryStream(result);
            stream.Position = 0;
            
            byte[] header = CreateHeader();
            stream.Write(header, 0, header.Length);

            //-------------------------------------

            if ((HasChild) && (childOutput != null))
                stream.Write(childOutput, 0, childOutput.Length);

            return result;
        }


        /// <summary>
        /// Start AnonMapping Protocol
        /// </summary>
        public override void Anonymize()
        {
            #region AnonMapping TCP Header
            int ___temp__localPort = 0;
            int ___temp__remotePort = 0;
            int ___temp__sourcePort = 0;
            int ___temp__destinationPort = 0;
            object ___temp__sequenceN = null;
            object ___temp__ackN = null;
            object ___temp__flags = null;
            object ___temp__windowSize = null;
            object ___temp__urgentPointer = null;
            object ___temp__hOptions = null;

            //
            // AnonMapping
            //
            Anonymizator anonymizator = new Anonymizator();
            ___temp__localPort =       packet.AnonSettings.TCP.Local.Run(Port);
            ___temp__remotePort =      packet.AnonSettings.TCP.Remote.Run(RemotePort);
            ___temp__sourcePort =      packet.AnonSettings.TCP.Source.Run(SourcePort);
            ___temp__destinationPort = packet.AnonSettings.TCP.Destination.Run(DestinationPort);
            Port = packet.AnonSettings.TCP.Local_Mask.Proceed(                 Port.ConvertPortNumberToBytesArray(),            ___temp__localPort.ConvertPortNumberToBytesArray()).ConvertToPortNumber();
            RemotePort = packet.AnonSettings.TCP.Remote_Mask.Proceed(          RemotePort.ConvertPortNumberToBytesArray(),      ___temp__remotePort.ConvertPortNumberToBytesArray()).ConvertToPortNumber();
            SourcePort = packet.AnonSettings.TCP.Source_Mask.Proceed(          SourcePort.ConvertPortNumberToBytesArray(),      ___temp__sourcePort.ConvertPortNumberToBytesArray()).ConvertToPortNumber();
            DestinationPort = packet.AnonSettings.TCP.Destination_Mask.Proceed(DestinationPort.ConvertPortNumberToBytesArray(), ___temp__destinationPort.ConvertPortNumberToBytesArray()).ConvertToPortNumber();


            anonymizator.AnonMap(sequenceN, out ___temp__sequenceN, packet.AnonSettings.TCP.SequenceNumber, packet.AnonSettings.TCP.SequenceNumber_ReplaceWith, AnonimizatorInput.DWORD);
            anonymizator.AnonMap(acknowledgmentN, out ___temp__ackN, packet.AnonSettings.TCP.AcknowledgmentNumber, packet.AnonSettings.TCP.AcknowledgmentNumber_ReplaceWith, AnonimizatorInput.DWORD);
            int flags = (((__cb_ns) ? 0x1 : 0x0) << 8) | 
                (((__cb_cwr) ? 0x1 : 0x0) << 7) | 
                (((__cb_ece) ? 0x1 : 0x0) << 6) | 
                (((__cb_urg) ? 0x1 : 0x0) << 5) | 
                (((__cb_ack) ? 0x1 : 0x0) << 4) | 
                (((__cb_psh) ? 0x1 : 0x0) << 3) | 
                (((__cb_rst) ? 0x1 : 0x0) << 2) | 
                (((__cb_syn) ? 0x1 : 0x0) << 1) | 
                ((__cb_fin) ? 0x1 : 0x0);
            anonymizator.AnonMap(flags,         out ___temp__flags,         packet.AnonSettings.TCP.Flags,         packet.AnonSettings.TCP.Flags_ReplaceWith,         AnonimizatorInput.WORD);
            anonymizator.AnonMap(windowSize,    out ___temp__windowSize,    packet.AnonSettings.TCP.WindowSize,    packet.AnonSettings.TCP.WindowSize_ReplaceWith,    AnonimizatorInput.WORD);
            anonymizator.AnonMap(urgentPointer, out ___temp__urgentPointer, packet.AnonSettings.TCP.UrgentPointer, packet.AnonSettings.TCP.UrgentPointer_ReplaceWith, AnonimizatorInput.WORD);
            anonymizator.AnonMap(hOptions,      out ___temp__hOptions,      packet.AnonSettings.TCP.HeaderOptions, packet.AnonSettings.TCP.HeaderOptions_ReplaceWith, hOptions.Length);





            if ((!packet.AnonSettings.TCP.Local.IsEmpty) || (!packet.AnonSettings.TCP.Remote.IsEmpty))
            {
                Port = (int)___temp__localPort;
                RemotePort = (int)___temp__remotePort;
            }
            if ((!packet.AnonSettings.TCP.Source.IsEmpty) || (!packet.AnonSettings.TCP.Destination.IsEmpty))
            {
                SourcePort = (int)___temp__sourcePort;
                DestinationPort = (int)___temp__destinationPort;
            }

            sequenceN = (uint)___temp__sequenceN;
            acknowledgmentN = (uint)___temp__ackN;
            __cb_ns  = (((flags & 0x100) >> 8) == 0x1) ? true : false;
            __cb_cwr = (((flags & 0x80) >> 7) == 0x1) ? true : false;
            __cb_ece = (((flags & 0x40) >> 6) == 0x1) ? true : false;
            __cb_urg = (((flags & 0x20) >> 5) == 0x1) ? true : false;
            __cb_ack = (((flags & 0x10) >> 4) == 0x1) ? true : false;
            __cb_psh = (((flags & 0x08) >> 3) == 0x1) ? true : false;
            __cb_rst = (((flags & 0x04) >> 2) == 0x1) ? true : false;
            __cb_syn = (((flags & 0x02) >> 1) == 0x1) ? true : false;
            __cb_fin = ((flags & 0x1) == 0x1) ? true : false;
            windowSize = (int)___temp__windowSize;
            urgentPointer = (int)___temp__urgentPointer;
            hOptions = (byte[])___temp__hOptions;

            //
            // Check TCP Header
            //


            #endregion


            if (HasChild)
            {
                if (packet.AnonSettings.TCP.Content == AnonContentType.Standard) // default settings
                    child.Anonymize();
                else
                {
                    byte[] content = child.Save();
                    object contentOut = new byte[content.Length];
                    anonymizator.AnonMap(content, out contentOut, packet.AnonSettings.TCP.Content.ConvertToAnonymizationType(), packet.AnonSettings.TCP.Content_ReplaceWith, content.Length);
                    content = (byte[])contentOut;
                    child = new Protocol(content, packet);
                }
            }

            
        }


        /// <summary>
        /// Get Full Packet Information for Advanced Packet Viewing
        /// </summary>
        /// <returns>Specialized Information Object</returns>
        public override object GetFullInformation()
        {
            IList<object> result = new List<object>();
            IDictionary<string, object> browserInfo = new Dictionary<string, object>();
            IList<KeyValuePair<Point, object>> browserHex = new List<KeyValuePair<Point, object>>();
            IList<string> browserGroup = new List<string>();
            result.Add(browserInfo);
            result.Add(browserHex);
            result.Add(browserGroup);

            #region __GET NAMEs
            {
                string name = String.Format("Transmission Control Protocol, Src Port: {0} ({1}), Dst Port: {2} ({3}), Seq: {4}, Ack: {5}",
                    ((SourcePort.GetPortName(ProtocolType.TCP) == "") ? SourcePort.ToString() : SourcePort.GetPortName(ProtocolType.TCP)),
                    SourcePort,
                    ((DestinationPort.GetPortName(ProtocolType.TCP) == "") ? DestinationPort.ToString() : DestinationPort.GetPortName(ProtocolType.TCP)),
                    DestinationPort,
                    sequenceN,
                    acknowledgmentN);
                if (HasChild)
                    name += String.Format(", Len: {0}", child.TotalLength);

                string item1 = String.Format("Source Port: {0} ({1})",
                    ((SourcePort.GetPortName(ProtocolType.TCP) == "") ? SourcePort.ToString() : SourcePort.GetPortName(ProtocolType.TCP)), SourcePort);
                string item2 = String.Format("Destination Port: {0} ({1})",
                                ((DestinationPort.GetPortName(ProtocolType.TCP) == "") ? DestinationPort.ToString() : DestinationPort.GetPortName(ProtocolType.TCP)), DestinationPort);

                string item3 = String.Format("Sequence number: {0}", sequenceN);
                string item4 = String.Format("Acknowledgment number: {0}", acknowledgmentN);
                string item5 = String.Format("Header length: {0}", ((long)IHL).ByteSize());
                int flags1 = (Flags & 0xF00) >> 8;
                int flags2 = (Flags & 0x0F0) >> 4;
                int flags3 = Flags & 0x00F;
                string item6 = String.Format("Flags: 0x{0}{1}{2}", flags1, flags2, flags3);
                item6 += String.Format("{0}", GetFlagsInformation());
                IDictionary<string, object> item6Collection = new Dictionary<string, object>();
                {
                    string item61 = String.Format("000. .... .... = Reserved : Not set");
                    string item62 = String.Format("...{0} .... .... = Nonce: {1}", __cb_ns ? 1 : 0, __cb_ns ? "Set" : "Not set");
                    string item63 = String.Format(".... {0}... .... = Congestion Window Reduced (CWR): {1}", __cb_cwr ? 1 : 0, __cb_cwr ? "Set" : "Not set");
                    string item64 = String.Format(".... .{0}.. .... = ECN-Echo: {1}", __cb_ece ? 1 : 0, __cb_ece ? "Set" : "Not set");
                    string item65 = String.Format(".... ..{0}. .... = Urgent: {1}", __cb_urg ? 1 : 0, __cb_urg ? "Set" : "Not set");
                    string item66 = String.Format(".... ...{0} .... = Acknowledgment: {1}", __cb_ack ? 1 : 0, __cb_ack ? "Set" : "Not set");
                    string item67 = String.Format(".... .... {0}... = Push: {1}", __cb_psh ? 1 : 0, __cb_psh ? "Set" : "Not set");
                    string item68 = String.Format(".... .... .{0}.. = Reset: {1}", __cb_rst ? 1 : 0, __cb_rst ? "Set" : "Not set");
                    string item69 = String.Format(".... .... ..{0}. = Syn: {1}", __cb_syn ? 1 : 0, __cb_syn ? "Set" : "Not set");
                    string item6A = String.Format(".... .... ...{0} = Fin: {1}", __cb_fin ? 1 : 0, __cb_fin ? "Set" : "Not set");
                    item6Collection[item61] = null;
                    item6Collection[item62] = null;
                    item6Collection[item63] = null;
                    item6Collection[item64] = null;
                    item6Collection[item65] = null;
                    item6Collection[item66] = null;
                    item6Collection[item67] = null;
                    item6Collection[item68] = null;
                    item6Collection[item69] = null;
                    item6Collection[item6A] = null;
                }
                string item7 = String.Format("Window size: {0}", windowSize);
                string checksumString = "";
                string checksumResult = "[Verification Not Implemented Yet]";
                {
                    int left = (checksum & 0xFF00) >> 8;
                    int right = (checksum & 0x00FF);
                    string leftString = String.Format("{0:X}", left);
                    if (leftString.Length == 1)
                        leftString = "0" + leftString;
                    string rightString = String.Format("{0:X}", right);
                    if (rightString.Length == 1)
                        rightString = "0" + rightString;
                    checksumString = "0x" + leftString + rightString;
                }
                string item8 = String.Format("Checksum: {0} {1}", checksumString, checksumResult);
                string item9 = String.Format("Options: ({0})", ((long)hOptions.Length).ByteSize());
                IDictionary<string, object> inner = new Dictionary<string, object>();
                inner[item1] = null;
                inner[item2] = null;
                inner[item3] = null;
                inner[item4] = null;
                inner[item5] = null;
                inner[item6] = item6Collection;
                inner[item7] = null;
                inner[item8] = null;
                inner[item9] = null;
                browserInfo[name] = inner;
            }
            #endregion

            #region __GET HEX
            {
                Point nameHex = new Point(1, IHL);
                Point item1Hex = new Point(1, 2);
                Point item2Hex = new Point(3, 4);
                Point item3Hex = new Point(5, 8);
                Point item4Hex = new Point(9, 12);
                Point item5Hex = new Point(13, 13);
                Point item6Hex = new Point(13, 14);
                IList<KeyValuePair<Point, object>> item6HexCollection = new List<KeyValuePair<Point, object>>();
                {
                    Point item61Hex = new Point(13, 13);
                    Point item63Hex = new Point(14, 14);
                    for (int i = 1; i <= 2; i++)
                        item6HexCollection.Add(new KeyValuePair<Point, object>(item61Hex, null));
                    for (int i = 1; i <= 8; i++)
                        item6HexCollection.Add(new KeyValuePair<Point, object>(item63Hex, null));
                }
                Point item7Hex = new Point(15, 16);
                Point item8Hex = new Point(17, 18);
                Point item9Hex = new Point(19, 19 + hOptions.Length - 1);

                IList<KeyValuePair<Point, object>> innerHex = new List<KeyValuePair<Point, object>>();
                innerHex.Add(new KeyValuePair<Point, object>(item1Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item2Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item3Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item4Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item5Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item6Hex, item6HexCollection));
                innerHex.Add(new KeyValuePair<Point, object>(item7Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item8Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item9Hex, null));
                browserHex.Add(new KeyValuePair<Point, object>(nameHex, innerHex));
            }
            #endregion

            #region __GET GROUP
            {
                browserGroup.Add(shortName);
            }
            #endregion

            return result;
        }




        



    }
}
