﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Linq;
using PacMap.Anonymize.Transactions;

namespace PacMap.Anonymize.HTTPRotator
{
    /// <summary>
    /// HTRI - HTTP Text Rotator Item
    /// </summary>
    public class RotatorItem
    {
        /// <summary>
        /// Creates Empty HTRI
        /// </summary>
        public RotatorItem()
        {
            isCorrect = true;
            isEmpty = true;
        }


        /****************************************************************************************/
        // :: DATA FIELDs ::
        private bool isEmpty = false;
        private bool isCorrect = false;
        private bool isOverriden = false;
        private AnonTransaction overridingObject = null;

        private string collectionTarget = "";
        private string whatToReplace = "";
        private string replacingContent = "";

        private bool replaceOnlyIfFindedExactExp = false;
        private bool replaceAlways = false;



        /****************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Collection Target
        /// </summary>
        public string CollectionTarget
        {
            get
            {
                return collectionTarget;
            }
        }

        /// <summary>
        /// What to be found and replaced
        /// </summary>
        public string WhatToBeReplaced
        {
            get
            {
                return whatToReplace;
            }
        }

        /// <summary>
        /// Replacing content
        /// </summary>
        public string ReplacingContent
        {
            get
            {
                return replacingContent;
            }
        }

        /// <summary>
        /// If Replace only if finding term is equal to found term 
        /// </summary>
        public bool ReplaceOnlyIfFindedExactExpression
        {
            get
            {
                return replaceOnlyIfFindedExactExp;
            }
        }

        /// <summary>
        /// If Replace Always is activated
        /// </summary>
        public bool ReplaceAlways
        {
            get
            {
                return replaceAlways;
            }
        }

        /// <summary>
        /// HTTP Rotator is in correct state
        /// </summary>
        public bool IsCorrect
        {
            get
            {
                return isCorrect;
            }
        }

        /// <summary>
        /// If HTRI contains any transaction
        /// </summary>
        public bool ContainsTransaction
        {
            get
            {
                return isOverriden;
            }
        }

        /// <summary>
        /// Get Containing Transaction (if it is supported)
        /// </summary>
        public AnonTransaction Transaction
        {
            get
            {
                return overridingObject;
            }
        }
        #endregion


        /****************************************************************************************/
        // :: GLOBAL METHODs ::

        

        /// <summary>
        /// Tries to Parse user "Phrase" - Possibly Creates Brand New HTRI Unit
        /// </summary>
        /// <param name="phrase">user "Phrase" - HINT : ("where:what#bywhat") or ("where:'whatexactly'#bywhat") or for always replacing ("where:''#bywhat")</param>
        /// <param name="result">new HTRI</param>
        /// <returns>if all goes well</returns>
        public static bool TryParse(string phrase, out RotatorItem result)
        {
            bool bResult = false;
            result = Create(phrase);
            if (result.IsCorrect)
            {
                bResult = true;
            }


            return bResult;
        }
        

        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>exporting string</returns>
        public override string ToString()
        {
            if (!isEmpty)
            {
                string result = "";
                string what = whatToReplace;
                if (replaceOnlyIfFindedExactExp)
                    what = "'" + what + "'";

                if (!isOverriden)
                    result = String.Format("{0}:{1}#{2}", collectionTarget, what, replacingContent);
                else
                    result = String.Format("{0}:{1}#!{2}", collectionTarget, what, overridingObject.ToString());
                return result;
            }
            return "none";
        }

        
        /// <summary>
        /// Create Brand New HTRI in correct state (not for user)
        /// </summary>
        /// <returns></returns>
        public static RotatorItem Create(string ct, string wtbr, string rc, bool exactexpression)
        {
            RotatorItem result = new RotatorItem();
            result.isEmpty = false;

            if ((ct != "") && (rc != ""))
            {
                result.isCorrect = true;
                if (wtbr == "")
                    result.replaceAlways = true;

                result.collectionTarget = ct;
                result.whatToReplace = wtbr;
                result.replacingContent = rc;
                result.replaceOnlyIfFindedExactExp = exactexpression;
            }

            return result;
        }


        /// <summary>
        /// Create Brand New HTRI from String Phrase (not for user)
        /// </summary>
        /// <param name="phrase">user Phrase</param>
        /// <returns>new HTRI in correct state, if phrase was parsed successfully</returns>
        public static RotatorItem Create(string phrase)
        {
            RotatorItem result = new RotatorItem();
            result.isEmpty = false;

            // Parse Phrase Ordinally
            string[] groups = phrase.Split(new char[] { ':', '#' }, StringSplitOptions.RemoveEmptyEntries);
            if (groups.Count() > 3)
            {
                string concating = String.Join(":", groups);
                int lengthOfFirstTwo = groups[0].Length + groups[1].Length;
                concating = concating.Substring(lengthOfFirstTwo + 2, concating.Length - lengthOfFirstTwo - 2);
                groups = new List<string>() { groups[0], groups[1], concating }.ToArray();
            }
            if (groups.Count() == 3)
            {
                char leftSep = phrase[groups[0].Length];
                char rightSep = phrase[groups[0].Length + groups[1].Length + 1];
                if ((leftSep == ':') && (rightSep == '#'))
                {
                    string target = groups[0];
                    string what = groups[1];
                    string content = groups[2];
                    bool findingExact = false;
                    if (((what[0] == '"') && (what[what.Length - 1] == '"')) ||
                        ((what[0] == '\'') && (what[what.Length - 1] == '\'')))
                    {
                        findingExact = true;
                        what = what.Substring(1, what.Length - 2);
                        if (what == "")
                            result.replaceAlways = true;
                    }

                    result = Create(target, what, content, findingExact);
                }
                else
                    result.isCorrect = false;
            }
            else
            {
                result.isCorrect = false;
            }



            // Test if overriding is supported
            if (result.IsCorrect)
            {
                AnonTransaction transaction = null;
                bool isOverrideable = TestForOverridingTransaction(result.replacingContent, out transaction);
                if (isOverrideable)
                {
                    result.isEmpty = false;
                    result.isCorrect = true;
                    result.replacingContent = "";
                    
                    result.isOverriden = true;
                    result.overridingObject = transaction;
                }
            }
            

            return result;
        }


        /****************************************************************************************/
        // :: FUNCTIONs ::

        /// <summary>
        /// Test for any AnonMapping Overriding Transaction
        /// </summary>
        /// <param name="bywhat">bywhat user part</param>
        /// <param name="transaction">new transaction, if supported</param>
        /// <returns>if Overriding is Supported</returns>
        private static bool TestForOverridingTransaction(string bywhat, out AnonTransaction transaction)
        {
            bool bResult = false;
            transaction = null;

                    if (bywhat.Length > 1)
                    {
                        if (bywhat[0] == '!')
                        {
                            string action = bywhat.Substring(1, bywhat.Length - 1);
                            if (action.Length > 5)
                            {
                                string cmd = action.Substring(0, 3);
                                string left = action.Substring(3, 1);
                                string right = action.Substring(action.Length - 1, 1);
                                //string[] values = action.Split(new char[] { '(', ')' }, StringSplitOptions.RemoveEmptyEntries);

                                if ((left == "(") && (right == ")"))
                                {
                                    IList<string> values = new List<string>();
                                    values.Add(cmd);
                                    values.Add(action.Substring(4, action.Length - 5));

                                    if (values[0] == "AVT")
                                    {
                                        bResult = true;
                                        transaction = new AnonValueTransaction(values[1]); // but maximal value is default 65535
                                    }
                                    if (values[0] == "ADT")
                                    {
                                        bResult = true;
                                        transaction = new AnonDateTransaction(values[1]);
                                    }
                                    if (values[0] == "AST")
                                    {
                                        bResult = true;
                                        transaction = new AnonStringTransaction(values[1]);
                                    }
                                }
                            }
                        }            
            }



            // Test if transaction is correct
            if (bResult)
            {
                if (!transaction.IsCorrect)
                {
                    bResult = false;
                    transaction = null;
                }
            }



            return bResult;
        }


    }
}
