﻿//*********************\\
//* HashFunction class
//* ver 1.00
//*********************\\
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Security.Cryptography;

namespace PacMap.Anonymize
{
    /// <summary>
    /// HFO (Hashing Function Object)
    /// </summary>
    public class HashFunction
    {
        #region Constructors
        /// <summary>
        /// Constructs Default HFO
        /// </summary>
        public HashFunction()
        {
            type = HashType.User;
            ParseUserOperations();
        }


        /// <summary>
        /// Constructs User HFO
        /// </summary>
        /// <param name="userAction">user hashing operation to make</param>
        public HashFunction(string userAction)
        {
            type = HashType.User;
            if (userAction != null)
            {
                if (userAction != "")
                    actionDiscription = userAction;
            }
            ParseUserOperations();
        }

        /// <summary>
        /// Constructs Advanced HFO
        /// </summary>
        /// <param name="kind">hash function type</param>
        /// <param name="userAction"></param>
        public HashFunction(HashType kind, string userAction)
        {
            type = kind;
            if (type == HashType.User)
            {
                if (userAction != null)
                {
                    if (userAction != "")
                        actionDiscription = userAction;
                }
                ParseUserOperations();
            }
        }
        #endregion

        /*****************************************************************************************/
        // :: DATA FIELDs ::
        delegate int Operation(int input);
        private HashType type = HashType.User;

        /// <summary>
        /// User Action Discription (Text)
        /// </summary>
        private string actionDiscription = "default"; // examples: "*2%50+50" || "/5+8%128+60"        

        /// <summary>
        /// User Action Variable
        /// </summary>
        private Operation action = null;    

        /*****************************************************************************************/
        // :: GLOBAL METHODs ::
        /// <summary>
        /// General Method For Start Hashing Array of Bytes
        /// </summary>
        /// <param name="input">input array</param>
        /// <returns>output array</returns>
        public byte[] HashThis(byte[] input)
        {
            switch (type)
            {
                case HashType.User:
                    return UserHash(input);           // run inner user hashing function method
                case HashType.MD5:
                case HashType.SHA1:
                    return KnownAlgorythmHash(input); // run inner wellknown hashing function method
                default: // it never happens
                    return null;
            }
        }


        /*****************************************************************************************/
        // :: PRIVATE HELPING FUNCTIONs::

        /// <summary>
        /// Parse USER's string 'actionDiscription' --> Action
        /// </summary>
        private void ParseUserOperations()
        {
            bool anyError = false;

            char[] numseps = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ' ' }; // numeric separators
            char[] opeseps = { '+', '-', '*', '/', '%', ' ' };                          // operations separators

            string[] operations = actionDiscription.Split(numseps, StringSplitOptions.RemoveEmptyEntries); // operations
            string[] numerics = actionDiscription.Split(opeseps, StringSplitOptions.RemoveEmptyEntries);   // string numbers
            #region int[] numbers = (int[])numerics;
            int[] numbers = null; // numbers
            {
                IList<int> numberList = new List<int>();
                foreach (var numeric in numerics)
                {
                    int i;
                    if (!int.TryParse(numeric, out i))
                        anyError = true;
                    numberList.Add(i);
                }
                numbers = numberList.ToArray();
            }
            #endregion



            if (operations.Count() != numerics.Count()) // if operators count and numbers are not equal, then bad user actionDiscription was made
                anyError = true;
            else // if looks like well,...
            {
                foreach (var operation in operations)   // ...check every expression for operator identifier presence
                {
                    if ((operation != "+") && (operation != "-") && (operation != "*") && (operation != "/") && (operation != "%")) // check for operator identifiers
                        anyError = true;
                }
            }

            if (!anyError) // if no error
            {
                action = new Operation((int input) =>
                {
                    #region Operations
                    Operation add = new Operation((int num) =>
                    {
                        return input + num;
                    });

                    Operation sub = new Operation((int num) =>
                    {
                        return input - num;
                    });

                    Operation mul = new Operation((int num) =>
                    {
                        return input * num;
                    });

                    Operation dev = new Operation((int num) =>
                    {
                        return input / num;
                    });

                    Operation mod = new Operation((int num) =>
                    {
                        return input % num;
                    });
                    #endregion

                    for (int i = 0; i < operations.Length; i++)
                    {
                        if (operations[i] == "+")
                            input = add(numbers[i]);
                        if (operations[i] == "-")
                            input = sub(numbers[i]);
                        if (operations[i] == "*")
                            input = mul(numbers[i]);
                        if (operations[i] == "/")
                            input = dev(numbers[i]);
                        if (operations[i] == "%")
                            input = mod(numbers[i]);
                    }

                    return input;
                });
            }
            else // if some error(s) appear(s)
            {
                action = new Operation((int num) =>
                {
                    return ((num.GetHashCode() * 70 + 96) % 256) & 0xFF; // then create default action
                });

            }

        }
                     



        /// <summary>
        /// Default 1byte-Hash
        /// </summary>
        /// <param name="input">input bytes array</param>
        /// <returns>output bytes array</returns>
        private byte[] UserHash(byte[] input)
        {
            IList<byte> result = new List<byte>();

            foreach (var b in input)    // for every byte of input...
            {
                int hash = action(b);   // .. hash it
                hash = (hash & 0xFF);
                result.Add((byte)hash); // ... and add it into the result collection
            }

            return result.ToArray();
        }


        /// <summary>
        /// (MD5 || SHA1) Hash
        /// </summary>
        /// <param name="input">input bytes array</param>
        /// <returns>output bytes array</returns>
        private byte[] KnownAlgorythmHash(byte[] input)
        {
            IList<byte> result = new List<byte>();

            HashAlgorithm generator = null;
            if (type ==  HashType.MD5)
                generator = new HMACMD5();
            if (type == HashType.SHA1)
                generator = new SHA1Managed();

            generator.Initialize();
            byte[] bloque = generator.ComputeHash(input); // generator will hash whole byte array

            #region Fill whole output array with bloque
            int size = input.Length / bloque.Length; // it gets size of how many will hashing be proceed
            if (size != 0) // if at least one block (at least 16 bytes and more)
            {
                for (int i = 1; i <= size; i++)
                {
                    result = result.Concat(bloque).ToList(); // add to result collection bloque of hashed byte arrays
                }
            }
            int rest = input.Length % bloque.Length; // (the rest - means "count % 16")
            if (rest != 0)
            {
                IEnumerable<byte> subset = bloque.Reverse().Take(rest); // tak last "rest value" count of bytes ...
                result = result.Concat(subset).ToList();                //..and also add to the result collection => NOW the input and output byte array size will be the same
            }
            #endregion

            return result.ToArray();
        }

        


    }


    /// <summary>
    /// Hashing Function Type
    /// </summary>
    public enum HashType
    {
        User,
        MD5,
        SHA1
    }
}
