﻿//********************************************************************************************************************************************************************************************
// :: AnonDateTransaction Class - Anonmapping Date Transaction ::
//                                                    ver 1.00
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// XML tag : "date-action" =
//                           "D"                     - replace anything by fix date D
//                           "(D1;D2)"               - replace anything by random date from datespan <D1,D2>;                                   !Condition: D1 must be less or equal then D2
//                           "(D1;D2):D3"            - replace only dates from datespan <D1,D2> by fix date D3                                  !Condition: D1 <= D2
//                           "(D1;D2):(D3;D4)"       - replace only dates from datespan <D1,D2> by random date from unique datespan <D3,D4>     !Condition: D1 <= D2 && D3 <= D4
//
// where D, D1, D2, D3 & D4 are single unique date
//*********************************************************************************************************************************************************************************************
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;

namespace PacMap.Anonymize.Transactions
{
    /// <summary>
    /// Anonmapping Date Transaction Class
    /// </summary>
    public class AnonDateTransaction : AnonTransaction
    {

        /// <summary>
        /// Construct Empty Anonmapping Date Transaction
        /// </summary>
        public AnonDateTransaction()
            : base()
        {
        }

        /// <summary>
        /// Construct User Anonmapping Date Transaction
        /// </summary>
        /// <param name="phrase">input string</param>
        public AnonDateTransaction(string phrase)
            : base(phrase)
        {
            if (phrase == "none")
            {
                isCorrect = true;
                isEmpty = true;
                return;
            }
            
            isEmpty = false;

            bool isConditional = false;
            IList<string> groups = phrase.Split(new char[] { '(', ')' }, StringSplitOptions.RemoveEmptyEntries);
            if (groups.Any(item => item == ":"))
            {
                if (groups.Count == 3)
                {
                    groups = new List<string>() { "("+groups[0]+")", "("+groups[2]+")" };
                    isConditional = true;
                }
            }



            if (!isConditional)
            {
                #region == "D" ==
                // == "D" ==
                {
                    DateTime date;
                    bool dateOk = DateTime.TryParse(phrase, out date);
                    if (dateOk)
                    {
                        when = WhenReplacing.Always;
                        content = ReplacingContent.SingleDate;
                        span = new DateSpan(date, new DateTime(0));
                    }
                }
                #endregion

                #region == (D1;D2) ==
                // == (D1;D2) ==
                {
                    if (phrase.Length > 2)
                    {
                        string left = phrase.Substring(0, 1);
                        string right = phrase.Substring(phrase.Length - 1, 1);
                        string contentText = phrase.Substring(1, phrase.Length - 2);
                        string[] values = contentText.Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);
                        if ((left == "(") && (right == ")") && (values.Count() == 2))
                        {
                            DateTime l;
                            DateTime r;
                            bool lOk = DateTime.TryParse(values[0], out l);
                            bool rOk = DateTime.TryParse(values[1], out r);
                            if ((lOk) && (rOk) && (l.Ticks <= r.Ticks))
                            {
                                when = WhenReplacing.Always;
                                content = ReplacingContent.DateSpan;
                                span = new DateSpan(l, r);
                            }
                        }
                    }
                }
                #endregion
            }
            else
            {
                #region == (D1;D2):D3 ==
                // == (D1;D2):D3 ==
                {
                    if (groups[0].Length > 2)
                    {
                        string left = groups[0].Substring(0, 1);
                        string right = groups[0].Substring(groups[0].Length - 1, 1);
                        string contentText = groups[0].Substring(1, groups[0].Length - 2);
                        string[] values = contentText.Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);
                        if ((left == "(") && (right == ")") && (values.Count() == 2))
                        {
                            DateTime d1;
                            DateTime d2;
                            DateTime d3;
                            bool d1Ok = DateTime.TryParse(values[0], out d1);
                            bool d2Ok = DateTime.TryParse(values[1], out d2);
                            bool d3Ok = DateTime.TryParse(groups[1], out d3);
                            if ((d1Ok) && (d2Ok) && (d3Ok) && (d1.Ticks <= d2.Ticks))
                            {
                                when = WhenReplacing.OnlyIfInInterval;
                                content = ReplacingContent.SingleDate;
                                filter = new DateSpan(d1, d2);
                                span = new DateSpan(d3, new DateTime(0));
                            }
                        }
                    }
                }
                #endregion

                #region == (D1;D2):(D3;D4) ==
                // == (D1;D2):(D3;D4) ==
                {
                    if ((groups[0].Length > 2) && (groups[1].Length > 2))
                    {
                        string left1 = groups[0].Substring(0, 1);
                        string right1 = groups[0].Substring(groups[0].Length - 1, 1);
                        string contentText1 = groups[0].Substring(1, groups[0].Length - 2);
                        string[] values1 = contentText1.Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);

                        string left2 = groups[1].Substring(0, 1);
                        string right2 = groups[1].Substring(groups[1].Length - 1, 1);
                        string contentText2 = groups[1].Substring(1, groups[1].Length - 2);
                        string[] values2 = contentText2.Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);

                        if (((values1.Count() == 2) && (left1 == "(") && (right1 == ")")) &&
                            ((values2.Count() == 2) && (left2 == "(") && (right2 == ")")))
                        {
                            DateTime d1;
                            DateTime d2;
                            DateTime d3;
                            DateTime d4;
                            bool d1Ok = DateTime.TryParse(values1[0], out d1);
                            bool d2Ok = DateTime.TryParse(values1[1], out d2);
                            bool d3Ok = DateTime.TryParse(values2[0], out d3);
                            bool d4Ok = DateTime.TryParse(values2[1], out d4);
                            if ((d1Ok) && (d2Ok) && (d1.Ticks <= d2.Ticks) && (d3Ok) && (d4Ok) && (d3.Ticks <= d4.Ticks))
                            {
                                when = WhenReplacing.OnlyIfInInterval;
                                content = ReplacingContent.DateSpan;
                                filter = new DateSpan(d1, d2);
                                span = new DateSpan(d3, d4);
                            }
                        }

                    }
                }
                #endregion
            }


            if ((when != WhenReplacing.Unspecified) && (content != ReplacingContent.Unspecified))
            {
                isCorrect = true;
            }

        }


        /**********************************************************************************************/
        // :: DATA FIELDs ::
        private WhenReplacing when = WhenReplacing.Unspecified;
        private ReplacingContent content = ReplacingContent.Unspecified;
        private DateSpan filter = DateSpan.Empty;
        private DateSpan span = DateSpan.Empty;
        #region Data Structures
        private enum WhenReplacing
        {
            Always,
            OnlyIfInInterval,
            Unspecified
        }

        private enum ReplacingContent
        {
            SingleDate,
            DateSpan,
            Unspecified
        }
        private class DateSpan
        {
            public DateSpan(DateTime left, DateTime right)
            {
                X = left;
                Y = right;
            }
            public DateTime X { get; private set; }
            public DateTime Y { get; private set; }
            public static DateSpan Empty = new DateSpan(new DateTime(0), new DateTime(0));
        }
        #endregion

        /**********************************************************************************************/
        // :: PROPERTIEs ::

        /**********************************************************************************************/
        // :: FUNCTIONs ::
        /// <summary>
        /// Converge DateTime to HTTP Protocol Time Format
        /// </summary>
        /// <param name="time">input Time</param>
        /// <returns>output correct HTTP Protocol Time Format</returns>
        private string Converge(DateTime time)
        {
            IDictionary<DayOfWeek, string> days = new Dictionary<DayOfWeek, string>()
            {
                {DayOfWeek.Monday, "Mon"},
                {DayOfWeek.Tuesday, "Tue"},
                {DayOfWeek.Wednesday, "Wed"},
                {DayOfWeek.Thursday, "Thu"},
                {DayOfWeek.Friday, "Fri"},
                {DayOfWeek.Saturday, "Sat"},
                {DayOfWeek.Sunday, "Sun"},
            };

            IDictionary<int, string> months = new Dictionary<int, string>()
            {
                {1, "Jan"},
                {2, "Feb"},
                {3, "Mar"},
                {4, "Apr"},
                {5, "May"},
                {6, "Jun"},
                {7, "Jul"},
                {8, "Aug"},
                {9, "Sep"},
                {10, "Oct"},
                {11, "Nov"},
                {12, "Dec"},
            };

            DateTime datetime = time.ToUniversalTime();
            
            string day = days[datetime.DayOfWeek];
            string month = months[datetime.Month];

            string result = String.Format("{0}, {1:00} {2} {3:0000} {4:00}:{5:00}:{6:00} GMT", day, datetime.Day, month, datetime.Year, datetime.Hour, datetime.Minute, datetime.Second);

            return result;
        }

        /**********************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Run Current Transaction on selected input item
        /// </summary>
        /// <param name="input">input number</param>
        /// <returns>output number</returns>
        public string Run(string input)
        {
            if ((isEmpty) && (isCorrect))
                return input;

            if (!isCorrect)
                return input;
            else
            {

                string result = "";
                DateTime date;
                if (!DateTime.TryParse(input, out date))
                {
                    result = input;
                }
                else
                {
                    //
                    // Get if will by replacing
                    //
                    bool willReplace = false;
                    if (when == WhenReplacing.Always)
                        willReplace = true;
                    else if (when == WhenReplacing.OnlyIfInInterval)
                    {
                        if ((filter.X.Ticks <= date.Ticks) && (date.Ticks <= filter.Y.Ticks))
                        {
                            willReplace = true;
                        }
                    }



                    //
                    // then replace, if it is supported
                    //
                    if (willReplace)
                    {
                        if (content == ReplacingContent.SingleDate) // by static date
                        {
                            result = Converge(span.X);
                        }
                        else if (content == ReplacingContent.DateSpan) // by random date
                        {
                            Random randomizator = new Random();
                            long difference = (long)(((span.Y.Ticks - span.X.Ticks) / 100000000));
                            difference = (randomizator.Next((int)difference));
                            difference = difference * 100000000;

                            long value = span.X.Ticks + difference;
                            DateTime newTime = new DateTime(value);
                            result = Converge(newTime);
                        }
                    }

                }

                return result;
            }
        }


        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>string result</returns>
        public override string ToString()
        {
            return String.Format("ADT({0})", Command);
        }


    }
}
