﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel;
using PacMap.Tracer;
using System.IO;
using System.Drawing;

namespace PacMap.Packet
{
    /// <summary>
    /// PPMI - Packet Packager Manager Interface
    /// </summary>
    public interface IPacketPackager
    {



        /********************************************************************/
        // :: PROPERTIEs ::
        /// <summary>
        /// Progress Reporting Event
        /// </summary>
        event ProgressChangedEventHandler ProgressChanged;

        /// <summary>
        /// Get Packets Count
        /// </summary>
        int Count { get; }

        /// <summary>
        /// Get PacketPackager State
        /// </summary>
        AppStatus InnerState { get; }

        /// <summary>
        /// Get Packet Collection from PacketPackager object
        /// </summary>
        IEnumerable<Packet> Items { get; }

        /// <summary>
        /// Get Correct State of PacketPackager
        /// </summary>
        bool IsCorrect { get; }


        /********************************************************************/
        // :: PUBLIC METHODs ::

        /// <summary>
        /// Show Trace File Content
        /// </summary>
        void ShowTraceFileContent();

        /// <summary>
        /// Show Simple Information for Every Single Packet
        /// </summary>
        void ShowSimplePacketInfo();

        /// <summary>
        /// Show Information Extensions for Every Single Packet
        /// </summary>
        void ShowAdvancedPacketInfo();

        /// <summary>
        /// Print Trace File Content into FILE
        /// </summary>
        /// <param name="input">input fileinfo</param>
        /// <returns>TRUE == SUCCESS</returns>
        bool PrintTraceFileContent(FileInfo input);

        /// <summary>
        /// Print Simple Information for Every Single Packet
        /// </summary>
        /// <param name="input">input fileinfo</param>
        /// <returns>TRUE == SUCCESS</returns>
        bool PrintSimplePacketInfo(FileInfo input);

        /// <summary>
        /// Filter packet list in some span
        /// </summary>
        /// <param name="span">items span</param>
        /// <param name="newList">new filtered list</param>
        /// <returns>TRUE == SUCCESS</returns>
        bool Filter(Point span, out IList<Packet> newList);


        /// <summary>
        /// Start AnonMappings Process On Every Single Packet in PacketPackager Class
        /// </summary>
        /// <exception cref="PacketPackagerNotCorrectException">Packet Packager is not in correct state!</exception>
        /// <exception cref="PacketPackagerCrossThreadsException">Packaget Packeger is already running operation!</exception>
        bool Anonmap();

        /// <summary>
        /// Save All Packets into the PCAP trace file
        /// </summary>
        /// <param name="filename">full path of file</param>
        /// <returns>TRUE if all goes well</returns>
        /// <exception cref="PacketPackagerNotCorrectException">Packet Packager is not in correct state!</exception>
        /// <exception cref="PacketPackagerCrossThreadsException">Packaget Packeger is already running operation!</exception>
        bool Save(string filename);


        /// <summary>
        /// Load packets from PCAP trace file (+ veryfing existence)
        /// </summary>
        /// <param name="filename">PCAP filename to open</param>
        /// <returns>TRUE if all goes well</returns>
        /// <exception cref="PacketPackagerNotEmptyException">Packet Packager is not empty for additional loading file!</exception>
        /// <exception cref="PacketPackagerCrossThreadsException">Packaget Packeger is already running operation!</exception>
        bool LoadFile(string filename);


        /// <summary>
        /// Load packets from trace file (existence expected)
        /// </summary>
        /// <param name="filename">fileinfo structure of file</param>
        /// <returns>TRUE if all goes well</returns>
        /// <exception cref="PacketPackagerNotEmptyException">Packet Packager is not empty for additional loading file!</exception>
        /// <exception cref="PacketPackagerCrossThreadsException">Packaget Packeger is already running operation!</exception>
        bool LoadFile(FileInfo filename);



    }
}
