﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using PacMap.General;
using System.IO;

namespace PacMap.Additionals.Exporting
{
    public class Table
    {

        /// <summary>
        /// Constructs new empty table
        /// </summary>
        public Table()
        {
            titles = new List<string>();
            columnRatios = new List<double>();
            columnFixed = new List<bool>();
        }

        protected ConsoleColor foreColor = ConsoleColor.Gray;
        protected ConsoleColor backColor = ConsoleColor.Black;
        protected ConsoleColor lineColor = ConsoleColor.DarkGray;
        protected ConsoleColor titleColor = ConsoleColor.Yellow;
        protected IList<string> titles = null;
        protected IList<object> data = null;
        protected IList<double> columnRatios = null;
        protected IList<bool> columnFixed = null;
        protected IList<int> linesPos = null;
        protected IList<int> startPos = null;


        protected int columns = 0;
        protected int rows = 0;

        protected int maxWidth = 0;

        protected enum LineTypeEnum
        {
            Up,
            Middle,
            Down
        }

        #region PROPERTIEs
        /// <summary>
        /// Get Number of Columns
        /// </summary>
        public int Columns
        {
            get
            {
                return columns;
            }
        }

        /// <summary>
        /// Get Number of Rows
        /// </summary>
        public int Rows
        {
            get
            {
                return rows;
            }
        }

        /// <summary>
        /// Get or Set Max Width
        /// </summary>
        public int MaxWidth
        {
            get
            {
                return maxWidth;
            }
            set
            {
                maxWidth = value;
            }
        }
        #endregion

        /***************************************************************************************************/
        // :: PRIVATE METHODs ::
        /// <summary>
        /// Recount width
        /// </summary>
        private void RecountNewRatios()
        {
            columnRatios = new List<double>();
            double ratio = 1d / data.Count;
            for (int i = 1; i <= data.Count; i++)
            {
                columnRatios.Add(ratio);
            }
        }

        /// <summary>
        /// Count variable new width
        /// </summary>
        private void RecountVariableRatios()
        {

            double fixedSum = 0;
            int falseCount = 0;
            for (int i = 1; i <= columnFixed.Count; i++)
            {
                if (columnFixed[i - 1])
                    fixedSum += columnRatios[i - 1];
                else
                    falseCount++;
            }

            double newRatio = (1d - fixedSum) / falseCount;

            for (int i = 1; i <= columnFixed.Count; i++)
            {
                if (!columnFixed[i - 1])
                {
                    columnRatios[i - 1] = newRatio;
                }
            }
        }

        /// <summary>
        /// Prepare Blank String
        /// </summary>
        /// <param name="length">new length</param>
        /// <returns>new blank string of specific length</returns>
        protected string PrepareBlankString(int length)
        {
            string result = "";
            for (int i = 1; i <= length; i++)
                result += " ";
            return result;
        }

        /// <summary>
        /// Get Max Width of Table
        /// </summary>
        private void GetMaxWidth()
        {
            int maxLength = 0;

            linesPos = new List<int>();
            startPos = new List<int>();

            for (int i = 1; i <= data.Count; i++)
            {
                IList<string> column = data[i - 1] as IList<string>;
                if (column != null)
                {
                    int maxItemLength = column.Max(item => item.Length);
                    if (titles[i - 1].Length > maxItemLength)
                        maxItemLength = titles[i - 1].Length;
                    maxItemLength += 4;

                    maxLength += maxItemLength;

                    linesPos.Add(maxLength);
                    startPos.Add(maxLength + 1);
                }
            }

            maxWidth = maxLength;
        }

        protected void PrintInit(bool isPrinting)
        {
            if ((maxWidth == 0) || (isPrinting))
                GetMaxWidth(); // it is necessary to get 'maxWidth' for printing into the file (for no data losses)

            RecountVariableRatios();

            #region Get linesPos and startPos
            if (!isPrinting)
            {
                linesPos = new List<int>();
                startPos = new List<int>();

                double sum = 0;
                foreach (var columnWidth in columnRatios)
                {
                    sum += columnWidth;
                    int linePos = (int)(sum * (maxWidth));
                    linesPos.Add(linePos - 1);
                    startPos.Add(linePos);
                }
            }
            #endregion
        }


        /// <summary>
        /// Print Just Line
        /// </summary>
        /// <param name="def">default char</param>
        /// <param name="leftedge">left char</param>
        /// <param name="middleedge">middle char</param>
        /// <param name="rightedge">right char</param>
        private void PrintJustLine(TextWriter writer, string def, string leftedge, string middleedge, string rightedge)
        {
            Console.ForegroundColor = lineColor;
            Console.CursorLeft = 0;
            for (int i = 0; i <= maxWidth - 1; i++)
            {
                if (i == 0)
                    writer.Write(leftedge);
                else if (i == linesPos.Last())
                    writer.Write(rightedge);
                else if (linesPos.Any(num => i == num))
                    writer.Write(middleedge);
                else
                    writer.Write(def);
            }
            writer.WriteLine();
        }

        /// <summary>
        /// Print Just Line
        /// </summary>
        /// <param name="type">line type</param>
        private void PrintJustLine(TextWriter writer, LineTypeEnum type, bool isPrinting)
        {   
            if (type == LineTypeEnum.Up)
                PrintJustLine(writer, "─", "┌", "┬", "┐");
            else if (type == LineTypeEnum.Middle)
                PrintJustLine(writer, "─", "├", "┼", "┤");
            else if (type == LineTypeEnum.Down)
                PrintJustLine(writer, "─", "└", "┴", "┘");
        }

        /// <summary>
        /// Print Just Line
        /// </summary>
        /// <param name="type"></param>
        protected void PrintJustLine(TextWriter writer, LineTypeEnum type)
        {
            PrintJustLine(writer, type, false);
        }

        /// <summary>
        /// Make new line in Console Out
        /// </summary>
        protected void NewLine(TextWriter writer, bool isPrinting)
        {
            if (!isPrinting)
            {
                Console.CursorLeft = maxWidth;
            }
            writer.WriteLine();
        }

       


        /// <summary>
        /// Print one line in chart
        /// </summary>
        /// <param name="collection">line items</param>
        /// <param name="isTitleHeader">if this is title line</param>
        protected virtual void PrintOneLine(TextWriter writer, bool isPrinting, string defaultSeparator, IList<string> collection, bool isTitleHeader)
        {
            Console.CursorLeft = 0;
            writer.Write(defaultSeparator);

            for (int i = 1; i < titles.Count + 1; i++)
            {
                int newLength = 0;
                if (i != titles.Count)
                {
                    if (i != 1)
                        newLength = linesPos[i - 1] - startPos[i - 2];
                    else
                        newLength = linesPos[i - 1] - 1;
                }
                else
                    newLength = maxWidth - 1 - startPos[i - 2];
                newLength -= 2;
                if (newLength < 0)
                    newLength = 0;

                string text = collection[i - 1];
                if (text.GetLength() > newLength)
                {
                    text = text.ColoredSubstring(0, newLength);
                    if (text.GetLength() >= 4)
                    {
                        text = text.ColoredSubstring(0, text.GetLength() - 3);
                        text += "$08...";
                    }
                    else if (text.GetLength() >= 2)
                    {
                        text = text.Substring(0, 1);
                        for (int j = 1; j <= newLength - 1; j++)
                            text += ".";
                    }
                }
                else if (text.GetLength() < newLength)
                {
                    text = text + PrepareBlankString(newLength - text.GetLength());
                }




                if (!isPrinting)
                {
                    if (i != 1)
                        Console.CursorLeft = startPos[i - 2];
                    else
                        Console.CursorLeft = 1;
                    Console.CursorLeft += 0;

                    if (isTitleHeader)
                        Console.ForegroundColor = titleColor;
                    else
                        Console.ForegroundColor = foreColor;
                }


                string mainText = String.Format(" {0} ", text);
                if (!isPrinting)
                    ConsoleGraphics.ColoredWrite(mainText);
                else
                    writer.Write(mainText.CleanColors());


                if (!isPrinting)
                {
                    Console.CursorLeft = linesPos[i - 1];
                    Console.ForegroundColor = lineColor;
                }
                writer.Write(defaultSeparator);
            }
        }

        /// <summary>
        /// Print one line in char with specialized char between items (HELPING FUNCTION TO FILES)
        /// </summary>
        /// <param name="defaultSeparator">specialized char</param>
        /// <param name="collection">line items</param>
        protected virtual void PrintOneLine(TextWriter writer, string defaultSeparator, IList<string> collection)
        {
            for (int i = 1; i <= columns - 1; i++)
            {
                writer.Write("{0}{1}", collection[i - 1], defaultSeparator);
            }
            writer.Write("{0}", collection[columns - 1]);
        }

        
        /// <summary>
        /// !! MAIN METHOD !!
        /// Print table onto the Plain text format (STDOUT or FILE)
        /// </summary>
        protected virtual void Print(TextWriter writer, bool isPrinting)
        {

            
            
            PrintInit(isPrinting);

            #region Table Header
            PrintJustLine(writer, LineTypeEnum.Up);

            PrintOneLine(writer, isPrinting, "│", titles, true); // print titles in char
            NewLine(writer, isPrinting);

            PrintJustLine(writer, LineTypeEnum.Middle);
            #endregion

            #region Table Content
            for (int i = 1; i <= rows; i++)
            {
                IList<string> lineContent = new List<string>();
                foreach (var dataItem in data)
                {
                    IList<string> column = dataItem as IList<string>;
                    if (column != null)
                    {
                        lineContent.Add(column[i - 1]);
                    }
                }

                PrintOneLine(writer, isPrinting, "│", lineContent, false);
                NewLine(writer, isPrinting);

            }
            #endregion

            PrintJustLine(writer, LineTypeEnum.Down);
        }
        /***************************************************************************************************/
        // :: PUBLIC METHODs ::

        /// <summary>
        /// Add new columnt into the table
        /// </summary>
        /// <param name="name">title</param>
        /// <param name="data">collection of string</param>
        public void AddColumn(string name, IList<string> data)
        {
            if (this.data == null)
            {
                this.data = new List<object>();

                columns++;
                rows = data.Count;

                this.titles.Add(name);
                this.data.Add(data);
            }
            else
            {
                if (data.Count == rows)
                {
                    columns++;
                    this.titles.Add(name);
                    this.data.Add(data);
                }
            }

            RecountNewRatios();
            columnFixed.Add(false);
        }

        /// <summary>
        /// Chagne column width
        /// </summary>
        /// <param name="number">column number</param>
        /// <param name="ratio">new width in percentage</param>
        public void ChangeColumnWidth(int number, double ratio)
        {
            if ((number < 1) || (number > columnRatios.Count)) // bad number
            {
            }
            else
            {
                columnRatios[number - 1] = ratio;
                columnFixed[number - 1] = true;
            }
        }

        /// <summary>
        /// Delete Column at position
        /// </summary>
        /// <param name="number">column number</param>
        public void DeleteColumn(int number)
        {
            if ((number < 1) || (number > data.Count)) // bad number
            {
            }
            else
            {   
                data.RemoveAt(number - 1);
                columns--;
            }
        }

        /// <summary>
        /// Get column width
        /// </summary>
        /// <param name="number">column number</param>
        /// <returns>width</returns>
        public double GetColumnWidth(int number)
        {
            double result = 100;
            if ((number < 1) || (number > columnRatios.Count)) // bad number
            {
            }
            else
                result = columnRatios[number - 1];

            return result;
        }


        /// <summary>
        /// Print table onto the STDOUT
        /// </summary>
        public void Print()
        {
            Print(Console.Out, false);
        }
        


        /// <summary>
        /// Print table onto the FILE
        /// </summary>
        /// <param name="format">file-format</param>
        /// <param name="writer">output writer</param>
        public void Print(FileFormat format, TextWriter writer)
        {
            string defaultSeparator = "";
            if (format == FileFormat.ODT)
            {
                defaultSeparator = "\t";
            }
            if (format == FileFormat.CSV)
            {
                defaultSeparator = ";";
            }


            //
            // Formatting only with tabulators od semi-colons, without console layout
            //
            if ((format == FileFormat.ODT) || (format == FileFormat.CSV))
            {
                PrintOneLine(writer, defaultSeparator, titles);
                NewLine(writer, true);
                
                for (int i = 1; i <= rows; i++)
                {
                    IList<string> newCollection = new List<string>();
                    foreach (var dataItem in data)
                    {
                        IList<string> column = dataItem as IList<string>;
                        if (column != null)
                        {
                            newCollection.Add(column[i - 1]);
                        }
                    }

                    PrintOneLine(writer, defaultSeparator, newCollection);
                    NewLine(writer, true);
                }
            }

            //
            // Formatting only with console layout
            //
            if (format == FileFormat.TXT)
                Print(writer, true);
        }




    }
}
