﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace PacMap.General.Masks
{
    /// <summary>
    /// USER IPv4 Address MASK Class
    /// </summary>
    public class MaskIPv4Address : Mask
    {
        /// <summary>
        /// Construct Empty IPv4 Mask
        /// </summary>
        public MaskIPv4Address()
            : base()
        {

        }

        /// <summary>
        /// Construct User IPv4 Mask
        /// </summary>
        /// <param name="byteLength">total mask length</param>
        /// <param name="maskType">mask type</param>
        /// <param name="maskedString">mask string ('true' - original, 'false' - anonmapping)</param>
        public MaskIPv4Address(int byteLength, MaskEnum maskType, string maskedString)
            : base(byteLength, maskType, maskedString)
        {
        }


        /// <summary>
        /// Construct User IPv4 Mask from simple texted string (mandatory form: "XXX.X--.---.---") ('X' - original, '-' anonmapping)
        /// </summary>
        /// <param name="input">mask string</param>
        public MaskIPv4Address(string input)
        {
            if (input == "none")
            {
                isEmpty = true;
                isCorrect = true;
                return;
            }

            isEmpty = false;
            IList<string> groups = input.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries).ToList();
            if (groups.Count == 4)
            {
                if (groups.All(item => item.Length == 3))
                {
                    length = 4;
                    mask = new byte[length];

                    for (int i = 1; i <= 4; i++)
                    {
                        int item = 0;
                        for (int k = 1; k <= 3; k++)
                        {
                            if (groups[i - 1][k - 1] == 'X')
                            {
                                item = item | (0x01 << (3 - k));
                            }
                        }

                        mask[i - 1] = (byte)item;
                    }

                    isSpecialized = true;
                    isCorrect = true;
                }
                else
                {
                    isCorrect = false;
                    lastError = "Mask string has to be 12 character long \"---.---.---.---\"";
                }
            }
            else
            {
                isCorrect = false;
                lastError = "Mask string has to be 12 character long \"---.---.---.---\"";
            }
        }

        

        /**********************************************************************************************/
        // :: DATA FIELDs ::
        private bool isSpecialized = false;

        /**********************************************************************************************/
        // :: PROPERTIEs ::


        /**********************************************************************************************/
        // :: GLOBAL METHODs ::

        /// <summary>
        /// Proceed Mask on Input Item
        /// </summary>
        /// <param name="original">original data</param>
        /// <param name="anonmapped">anonmapped data</param>
        /// <returns>result string</returns>
        public override byte[] Proceed(byte[] original, byte[] anonmapped)
        {
            if ((IsEmpty) || (original.Length != anonmapped.Length) || (original.Length != mask.Length))
                return anonmapped;

            if (!isSpecialized)
                return base.Proceed(original, anonmapped);
            else
            {
                byte[] result = new byte[length];

                IList<string> originalString = original.ConvertIPv4ToString().Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries).ToList();
                IList<string> anonmappedString = anonmapped.ConvertIPv4ToString().Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries).ToList();

                IList<string> preresult = new List<string>();

                for (int i = 1; i <= 4; i++)
                {
                    string originalItem = originalString[i - 1];
                    if (originalItem.Length == 1)
                        originalItem = "00" + originalItem;
                    else if (originalItem.Length == 2)
                        originalItem = "0" + originalItem;
                    string anonmappedItem = anonmappedString[i - 1];
                    if (anonmappedItem.Length == 1)
                        anonmappedItem = "00" + anonmappedItem;
                    else if (anonmappedItem.Length == 2)
                        anonmappedItem = "0" + anonmappedItem;

                    string item = "";
                    for (int k = 1; k <= 3; k++)
                    {
                        if ((mask[i - 1] & (0x01 << (3 - k))) >> (3 - k) == 0x01)
                        {
                            item += originalItem[k - 1];
                        }
                        else
                            item += anonmappedItem[k - 1];
                    }

                    preresult.Add(item);
                }

                string nearlyDoneResult = string.Join(".", preresult);
                result = nearlyDoneResult.ConvertIPv4ToArray();

                return result;
            }
        }

        /// <summary>
        /// Convert To String
        /// </summary>
        /// <returns>string result</returns>
        public override string ToString()
        {
            if (!IsCorrect)
                return "badmask";
            if (IsEmpty)
                return "none";

            string result = "";
            if (!isSpecialized)
                result = base.ToString();
            else
            {
                IList<string> groups = new List<string>();
                for (int i = 1; i <= 4; i++)
                {

                    string item = "";
                    for (int k = 1; k <= 3; k++)
                    {
                        bool isMarked = (mask[i - 1] & (0x01 << (3 - k))) >> (3 - k) == 0x01 ? true : false;
                        if (isMarked)
                            item += "X";
                        else
                            item += "-";
                    }

                    groups.Add(item);
                }

                result = string.Join(".", groups);

            }

            return result;
        }

        /// <summary>
        /// Create New Mask from String
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>completed mask</returns>
        public static MaskIPv4Address Parse(string input)
        {
            MaskIPv4Address result = null;
            int byteLength;
            MaskEnum type;
            string maskedString;
            bool isCorrect = ParseBinHex(input, out byteLength, out type, out maskedString);

            
            if (!isCorrect)
            {
                result = new MaskIPv4Address(input);
                isCorrect = result.IsCorrect;
            }
            else
            {
                result = new MaskIPv4Address(byteLength, type, maskedString);
            }

            //if (isCorrect)
                return result;
            //else
            //{
            //    result;
            //}
        }



    }
}
