﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using PacMap.Packet;
using System.Drawing;

namespace PacMap.General
{
    /// <summary>
    /// General Extensions
    /// </summary>
    public static class GeneralExt
    {





        /// <summary>
        /// Make Line in Console
        /// </summary>
        public static void MakeConsoleLine()
        {
            MakeConsoleLine(false, false, null);
        }

        /// <summary>
        /// Make Line in Console
        /// </summary>
        /// <param name="isPrintingGenerally">Printing Enabled, Generally</param>
        /// <param name="isAlsoPrintingIntoFile">Manual Additional Printing</param>
        /// <param name="writer">output writer</param>
        public static void MakeConsoleLine(bool isPrintingGenerally, bool isAlsoPrintingIntoFile, TextWriter writer)
        {
            if (!isPrintingGenerally)
            {
                for (int i = 1; i <= Program.ConsoleMaxX - 1; i++)
                    ConsoleGraphics.ColoredWrite("$08─");
                Console.WriteLine();
            }
            else
            {
                if (isAlsoPrintingIntoFile)
                    writer.WriteLine("---");
            }
        }



        /// <summary>
        /// Parse HTTP Header Information
        /// </summary>
        /// <param name="collection">whole header</param>
        /// <param name="target">string to be searched</param>
        /// <returns>string result</returns>
        public static string ParseHeaderIntelligence(this string[] collection, string target)
        {
            string result = "";
            IEnumerable<string> lineChoosed = collection.Where(line => line.Split(':').First() == target); // choosed only line, where string before ':' is matching 'target'
            if (lineChoosed.Count() != 0)
            {
                foreach (var item in lineChoosed) // for every single result
                {
                    string innerResult = "";

                    #region int firstColonPosition = Get First Appearance of ':' in 'item'
                    int firstColonPosition = 0;
                    {
                        IList<int> positions = new List<int>();
                        int index = 0;
                        foreach (var ch in item)
                        {
                            if (ch == ':')
                            {
                                positions.Add(index);
                            }
                            index++;
                        }
                        firstColonPosition = positions.First();
                    }
                    #endregion
                    innerResult = item.Substring(firstColonPosition + 2, item.Length - firstColonPosition - 2); // get string after first ':'

                    if (result != "")
                        result += "\r\n";
                    result += innerResult;
                }
            }
            return result;
        }

        /// <summary>
        /// Save HTTP Header Information
        /// </summary>
        /// <param name="input">input name</param>
        /// <returns>all done</returns>
        public static IList<string> SaveHeaderIntelligence(this string[] collection, string target, string input, bool erase)
        {
            //
            // Erase origin data from collection
            //
            int insertingIndex = 0;
            int insertingIndexCounting = 0;
            
            if (erase)
            {
                IList<string> newColl = new List<string>();
                foreach (var line in collection)
                {
                    if (line.Split(new char[] { ':' }).First() == target)
                    {
                        insertingIndex = insertingIndexCounting;
                    }
                    else
                        newColl.Add(line);

                    insertingIndexCounting++;
                }
                collection = newColl.ToArray();
            }



            //
            // Add new info
            //
            string result = "";
            string[] groups = input.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);
            int whereIndex = 0;
            foreach (var item in groups)
            {
                string innerResult = "";

                innerResult = target + ": " + item;
                IList<string> newColl = collection.ToList();
                newColl.Insert(insertingIndex + whereIndex, innerResult);
                collection = newColl.ToArray();

                if (result != "")
                    result += "\r\n";
                result += innerResult;
                whereIndex++;
            }


            return collection;
        }








        /// <summary>
        /// Convert Port Number (0-65535) to WORD
        /// </summary>
        /// <param name="port">integer port</param>
        /// <returns>output word</returns>
        public static byte[] ConvertPortNumberToBytesArray(this int port)
        {
            byte[] result = new byte[2];
            result[0] = (byte)((port & 0xFF00) >> 8);
            result[1] = (byte)(port & 0x00FF);

            return result;
        }

        /// <summary>
        /// Convert WORD to Port Number (0-65535)
        /// </summary>
        /// <param name="word">input word</param>
        /// <returns>integer port</returns>
        public static int ConvertToPortNumber(this byte[] word)
        {
            int result = 0;

            result = word[0] << 8 | word[1];

            return result;
        }

        /// <summary>
        /// TryParse Procedure for parsing XML Anonmapping basic type
        /// </summary>
        /// <param name="input">input string</param>
        /// <param name="result">anonmapping value</param>
        /// <returns>TRUE = if success</returns>
        public static bool TryParse_AnonMappingValue(this string input, out Anonymize.AnonContentType result)
        {
            bool bResult = false;
            result = Anonymize.AnonContentType.None;
            if (input == "none")
            {
                bResult = true;
            }
            else if (input == "static")
            {
                bResult = true;
                result = Anonymize.AnonContentType.Static;
            }
            else if (input == "dynamic")
            {
                bResult = true;
                result = Anonymize.AnonContentType.Dynamic;
            }
            else if (input == "hash")
            {
                bResult = true;
                result = Anonymize.AnonContentType.Hash;
            }
            else if (input == "standard")
            {
                bResult = true;
                result = Anonymize.AnonContentType.Standard;
            }

            return bResult;
        }

        /// <summary>
        /// Move all hex positions to right with given seed addition 
        /// </summary>
        /// <param name="input">collection</param>
        /// <param name="plusPosition">seed addition number</param>
        /// <returns>edited collection</returns>
        public static IList<KeyValuePair<Point, object>> MoveHexPositions(this IList<KeyValuePair<Point, object>> input, int plusPosition)
        {
            IList<KeyValuePair<Point, object>> result = new List<KeyValuePair<Point, object>>();

            foreach (var item in input)
            {
                KeyValuePair<Point, object> inner = new KeyValuePair<Point, object>();
                Point newPoint = new Point(item.Key.X + plusPosition, item.Key.Y + plusPosition);

                if (item.Value == null)
                    inner = new KeyValuePair<Point, object>(newPoint, null);
                else
                    inner = new KeyValuePair<Point, object>(newPoint, MoveHexPositions((IList<KeyValuePair<Point, object>>)item.Value, plusPosition));

                result.Add(inner);
            }

            return result;
        }


        /// <summary>
        /// Convert To String
        /// </summary>
        /// <param name="input">general anonymization type</param>
        /// <returns>string result</returns>
        public static string ConvertToString(this Anonymize.AnonymizationType input)
        {
            return ConvertToString(input.ConvertToAnonInnerType());
        }

        /// <summary>
        /// Convert To String
        /// </summary>
        /// <param name="input">general inner anonymization type</param>
        /// <returns>string result</returns>
        public static string ConvertToString(this Anonymize.AnonContentType input)
        {
            string result = "";
            switch (input)
            {
                case Anonymize.AnonContentType.None:
                    result = "none";
                    break;
                case Anonymize.AnonContentType.Static:
                    result = "static";
                    break;
                case Anonymize.AnonContentType.Dynamic:
                    result = "dynamic";
                    break;
                case Anonymize.AnonContentType.Hash:
                    result = "hash";
                    break;
                case Anonymize.AnonContentType.Standard:
                    result = "standard";
                    break;
            }
            return result;
        }



        #region Bytes Converting & Processing
        /// <summary>
        /// Convert To One Hex String
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        public static string ConvertToOneHex(this byte input)
        {
            string result = String.Format("{0:X}", input);
            if (result.Length == 1)
                result = "0" + result;
            return result;
        }

        /// <summary>
        /// Convert String to Bytes Array
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>bytes collection/returns>
        public static IList<byte> ConvertStringToBytes(this string input)
        {
            IList<byte> result = new List<byte>();
            foreach (var item in input)
            {
                byte temp = (byte)item;
                result.Add(temp);
            }
            return result;
        }

        /// <summary>
        /// Converts HEX String to Byte[]
        /// </summary>
        /// <param name="hex">hex string</param>
        /// <returns>byte[] result</returns>
        public static byte[] ConvertHexStringToBytesArray(this string hex)
        {
            return Enumerable.Range(0, hex.Length)
                             .Where(x => x % 2 == 0)
                             .Select(x => Convert.ToByte(hex.Substring(x, 2), 16))
                             .ToArray();
        }

        /// <summary>
        /// Convert bytes array to texted collection
        /// </summary>
        /// <param name="input">bytes array</param>
        /// <returns>texted collection</returns>
        public static IList<string> ConvertToStringCollection(this byte[] input)
        {
            IList<string> result = new List<string>();

            string empty = "";
            foreach (var item in input)
            {
                if ((item != '\r') && (item != '\n') && (item != 0x07))
                {
                    empty = empty + String.Format("{0}", (char)item);
                }
                else if (item == 0x07)
                {
                    empty = empty + " ";
                }
                else if (item == '\n')
                {
                    result.Add(empty);
                    empty = "";
                }
            }
            return result;
        }


        public static string ConvertToString(this byte[] input)
        {
            string empty = "";
            foreach (var item in input)
            {
                if ((item != '\r') && (item != '\n') && (item != 0x07))
                {
                    empty = empty + String.Format("{0}", (char)item);
                }
                else if (item == 0x07)
                {
                    empty = empty + " ";
                }
                else if (item == '\n')
                {   
                    empty = "";
                }
            }

            return empty;
        }


        /// <summary>
        /// Get Size in string from numeric value
        /// </summary>
        /// <param name="input">numeric value</param>
        /// <returns>Size in string</returns>
        public static string ByteSize(this long input)
        {
            string result = "";

            string measure = "0";
            string unit = "B";


            if (input > 1000000000)
            {
                measure = String.Format("{0:0.00}", ((double)input) / 1000000000);
                unit = "GB";
            }
            else if (input > 1000000)
            {
                measure = String.Format("{0:0.00}", ((double)input) / 1000000);
                unit = "MB";
            }
            else if (input > 1000)
            {
                measure = String.Format("{0:0.00}", ((double)input) / 1000);
                unit = "kB";
            }
            else
            {
                measure = input.ToString();
                unit = "B";
            }

            result = measure + " " + unit;

            return result;
        }
        #endregion


        #region IP Converting
        /// <summary>
        /// Converts IPv6 (string --> byte[])
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>byte[] array</returns>
        public static byte[] ConvertIPv6ToArray(this string input)
        {
            byte[] result = new byte[16];
            string[] groups = input.Split(new char[] { ':' }, StringSplitOptions.RemoveEmptyEntries);
            if (groups.Count() != 8)
                return null;

            for (int j = 1; j <= 8; j++)
            {
                string wordString = groups[j-1];
                byte[] word = ConvertHexStringToBytesArray(wordString);
                result[(j - 1) * 2] = word[0];
                result[(j - 1) * 2 + 1] = word[1];
            }

            return result;
        }


        /// <summary>
        /// Converts IPv6 (byte[] --> string)
        /// </summary>
        /// <param name="input">byte[] input</param>
        /// <returns>string</returns>
        public static string ConvertIPv6ToString(this byte[] input)
        {
            string result = "";

            for (int i = 1; i <= 8; i++)
            {
                int j = (i - 1) * 2;
                if (result != "")
                    result += ":";

                string left = String.Format("{0:X}", input[j]);
                string right = String.Format("{0:X}", input[j + 1]);
                if (left.Length == 1)
                    left = "0" + left;
                if (right.Length == 1)
                    right = "0" + right;
                result += String.Format("{0}{1}", left, right);
            }

            return result;
        }
        


        /// <summary>
        /// Converts IPv4 (string --> byte[])
        /// </summary>
        /// <param name="input">string input</param>
        /// <returns>byte[] array</returns>
        public static byte[] ConvertIPv4ToArray(this string input)
        {
            byte[] result = new byte[4];
            char[] seps = { '.' };
            string[] groups = input.Split(seps, StringSplitOptions.RemoveEmptyEntries);
            for (int i = 0; i <= 3; i++)
            {
                int value;
                int.TryParse(groups[i], out value);
                result[i] = (byte)value;
            }

            return result;
        }



        /// <summary>
        /// Converts IPv4 (byte[] --> string)
        /// </summary>
        /// <param name="input">byte[] input</param>
        /// <returns>string</returns>
        public static string ConvertIPv4ToString(this byte[] input)
        {
            string result = ""; ;
            for (int i = 0; i <= 3; i++)
            {
                int value = input[i];
                if (result != "")
                    result += ".";
                result += value.ToString();
            }
            return result;
        }
        #endregion


       



       

       

        public static bool TestForOrdinalChars(this string input)
        {
            bool bresult = true;
            if (input.Any(ch => (int)ch < 0x20))
                bresult = false;

            return bresult;
        }





        public static Anonymize.AnonContentType ConvertToAnonInnerType(this Anonymize.AnonymizationType input)
        {
            switch (input)
            {
                case Anonymize.AnonymizationType.None:
                    return Anonymize.AnonContentType.None;
                case Anonymize.AnonymizationType.Static:
                    return Anonymize.AnonContentType.Static;
                case Anonymize.AnonymizationType.Dynamic:
                    return Anonymize.AnonContentType.Dynamic;
                case Anonymize.AnonymizationType.Hash:
                    return Anonymize.AnonContentType.Hash;
                default:
                    return Anonymize.AnonContentType.None;
            }
        }

        /// <summary>
        /// Convert to AnonymizationType enum
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        public static Anonymize.AnonymizationType ConvertToAnonymizationType(this Anonymize.AnonContentType input)
        {
            Anonymize.AnonymizationType result = Anonymize.AnonymizationType.None;
            switch (input)
            {
                case Anonymize.AnonContentType.None:
                case Anonymize.AnonContentType.Standard:
                    result = Anonymize.AnonymizationType.None;
                    break;
                case Anonymize.AnonContentType.Static:
                    result = Anonymize.AnonymizationType.Static;
                    break;
                case Anonymize.AnonContentType.Dynamic:
                    result = Anonymize.AnonymizationType.Dynamic;
                    break;
                case Anonymize.AnonContentType.Hash:
                    result = Anonymize.AnonymizationType.Hash;
                    break;
            }
            return result;
        }



        #region CMDLine Argument Checking
        /// <summary>
        /// Report errors on console
        /// </summary>
        /// <param name="errors"></param>
        public static void WriteErrors(this IEnumerable<string> errors)
        {
            foreach (var item in errors)
            {
                Console.WriteLine("!!{0}.", item);
            }
        }

        /// <summary>
        /// Argument Checking
        /// </summary>
        /// <param name="argument">arg</param>
        /// <returns>TRUE = if argument is command character</returns>
        public static bool ArgCheck(this string argument)
        {
            // Possible error
            if (argument == "")
                return false;

            if (argument.Substring(0, 1) != "-")
                return false;
            else
                return true;
        }


        /// <summary>
        /// Argument Checking
        /// </summary>
        /// <param name="argument">arg</param>
        /// <param name="expected">expected expression</param>
        /// <returns>TRUE = if argument correspondence with expected</returns>
        public static bool ArgCheck(this string argument, string expected)
        {
            // Possible error
            if ((argument == "") || (expected == ""))
                return false;

            // Non-command argument
            if ((argument.Substring(0, 1) != "-") && (argument.Substring(0,1) != "/"))
                return false;
            

            // Command argument
            return ArgCheck(argument, expected, expected.Substring(0, 1).ToLower(), 1);
        }

        /// <summary>
        /// Argument Checking
        /// </summary>
        /// <param name="argument">arg</param>
        /// <param name="expected">expected expression</param>
        /// <param name="expectedFirstLetter">expected first letter</param>
        /// <returns>TRUE = if argument correspondence with expected</returns>
        public static bool ArgCheck(this string argument, string expected, string expectedFirstLetter, int lengthofshortcut)
        {
            // Possible error
            if ((argument == "") || (expected == ""))
                return false;

            // Non-command argument
            if ((argument.Substring(0, 1) != "-") && (argument.Substring(0, 1) != "/"))
                return false;

            // Command argument
            string inner = argument.Substring(1, argument.Length - 1).ToLower();
            if ((inner.Length > 1) && (inner.Substring(0, 1) == "-"))
                inner = inner.Substring(1, inner.Length - 1);

            string alternative = inner;
            if (alternative.Length > lengthofshortcut)
                alternative = alternative.Substring(0, lengthofshortcut);


            return ((inner == expected) || ((alternative == expectedFirstLetter) && (inner.Length == lengthofshortcut)));
        }
        #endregion

    }
}
