﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using PacMap.General;
using System.Drawing;
using System.IO;

namespace PacMap.Packet
{
    /// <summary>
    /// PAIB (Packet Advanced Information Browser)
    /// </summary>
    public class PacketBrowser
    {

        /// <summary>
        /// Constructs PAIB - Packet Advanced Information Browser
        /// </summary>
        /// <param name="owner"></param>
        public PacketBrowser(PacketPackager owner)
        {
            if (!owner.IsCorrect)
            {
                AppSettings.AddNewError("onload packets are null?");
                isGood = false;
                return;
            }

            packager = owner;
            packets = packager.Items;
            packetsOriginal = packets.ToList();

            int index = 0;
            foreach (var packet in packetsOriginal)
	        {
		        index++;
                packetsNo[packet] = index;
	        }
            isGood = true;
        }


        /****************************************************************************************************/
        // :: DATA FIELDs ::
        private bool isGood = false;
        private PacketPackager packager = null;
        private IEnumerable<Packet> packets = null;
        private IList<Packet> packetsOriginal = null;
        private IList<Packet> packetsFiltered = null;
        private IList<Packet> packetsGrouping = null;
        private readonly IDictionary<Packet, int> packetsNo = new Dictionary<Packet, int>();
        private IList<string> packetsNames_Active = null;
        private IList<string> packetsNames_Deactive = null;
        private ConsoleColor window_Inactive_Forecolor = ConsoleColor.Black;
        private ConsoleColor window_Inactive_Backcolor = ConsoleColor.Gray;
        private Rectangle packetsWindow = Rectangle.Empty;
        private ConsoleColor packetsWindow_Forecolor = ConsoleColor.DarkGray;
        private ConsoleColor packetsWindow_Backcolor = ConsoleColor.Black;
        private ConsoleColor packetsWindow_Forecolor_HighLight = ConsoleColor.Gray;
        private ConsoleColor packetsWindow_Backcolor_HighLight = ConsoleColor.Yellow;
        private Rectangle infoWindow = Rectangle.Empty;
        private Rectangle searchWindow = Rectangle.Empty;
        private int searchWindowPosition = 0;
        private int searchWindowMinPosition = 0;
        private int searchWindowMaxPosition = 0;
        private ConsoleColor searchWindow_Forecolor = ConsoleColor.White;
        private ConsoleColor searchWindow_Backcolor = ConsoleColor.DarkMagenta;
        private ConsoleColor searchWindow_Forecolor_texted = ConsoleColor.Yellow;
        private ConsoleColor searchWindow_Forecolor_typed = ConsoleColor.DarkGray;
        private string searchWindowExpression = "";
        private Rectangle helpWindow = Rectangle.Empty;
        private ConsoleColor helpWindow_Forecolor = ConsoleColor.White;
        private ConsoleColor helpWindow_Backcolor = ConsoleColor.DarkGreen;
        private ConsoleColor helpWindow_Forecolor_texted = ConsoleColor.White;
        private ConsoleColor helpWindow_Forecolor_HighLight = ConsoleColor.Gray;
        private Rectangle packetWindow = Rectangle.Empty;
        private Rectangle hexWindow = Rectangle.Empty;
        private Rectangle protocolWindow = Rectangle.Empty;
        private ConsoleColor packetWindow_Forecolor = ConsoleColor.Gray;
        private ConsoleColor packetWindow_Backcolor = ConsoleColor.Black;
        private ConsoleColor hexWindow_Forecolor = ConsoleColor.White;
        private ConsoleColor hexWindow_Backcolor = ConsoleColor.DarkBlue;
        private ConsoleColor protocolWindow_Forecolor = ConsoleColor.White;
        private ConsoleColor protocolWindow_Backcolor = ConsoleColor.DarkMagenta;
        private int packetWindowIndex = 0;
        private int actualItem = 1;
        private int actualPage = 1;
        private int width = 79;
        private int height = 25;
        private int maxCountPacketPerPage = 0;
        private bool groupingEnabled = false;
        private bool filteringEnabled = false;
        private int filteringIndex = 0;
        private int filterWindowPosition = 0;
        private int filterWindowMinPosition = 0;
        private int filterWindowMaxPosition = 0;
        private string filterWindowExpression = "";
        private IDictionary<string, object> packetData_Info = null;
        private IList<KeyValuePair<Point, object>> packetData_Hex = null;
        private IList<string> packetData_Group = null;
        private byte[] packetData_Data = null;
        private IList<string> packetWindow_Data = null;
        private IList<Point> packetWindow_HexConnections = null;
        private IDictionary<int, int> packetWindow_EnterConnentions = null;
        private int protocolWindow_Actual = 0;
        private IList<string> hexWindow_Data = null;
        private int packetWindow_Scrolling = 0;
        private int packetWindow_Actual = 0;
        private int hexWindow_Scrolling = 0;
        private int protocolWindow_Scrolling = 0;
        private BrowserState browserState = BrowserState.MainMenu;
        delegate void InsertColorDelegate(int value, bool colorSwitch, bool activeSwitch);
        private enum BrowserState
        {
            MainMenu,
            Search,
            Search_Error,
            Help,
            Filter,
            Filter_Error,
            Info,
            Hex,
            Protocol,
            Loading
        }
        private readonly IDictionary<BrowserState, string> statusMessages = new Dictionary<BrowserState, string>()
        {
            {BrowserState.MainMenu, "$08▲,▼,◄,► $00Browse packets, $08└─> $00Select, $08ESC - $00Exit, $08H - $00Help"},
            {BrowserState.Loading, ""},
            {BrowserState.Search, "$080-9 $00Keys, $08└─> $00Select, $08 ESC - $00Cancel"},
            {BrowserState.Search_Error, "$08└─>/ESC - $00Cancel"},
            {BrowserState.Help, "$08└─>/ESC - $00Cancel"},
            {BrowserState.Filter, "$080-9 $00Keys, $08└─> $00Select, $08 ESC - $00Cancel"},
            {BrowserState.Filter_Error, "$08└─>/ESC - $00Cancel"},
            {BrowserState.Info, "$08▲,▼,└─> $00Browse information, $08TAB - $00Change window, $08ESC - $00Cancel"},
            {BrowserState.Hex, "$08▲,▼ $00Scroll up/down, $08TAB - $00Change window, $08ESC - $00Cancel"},
            {BrowserState.Protocol, "$08▲,▼ $00Select protocol, $08TAB - $00Change window, $08ESC - $00Cancel"},
        };

        /****************************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Get Total Pages
        /// </summary>
        public int Pages
        {
            get
            {
                int count = packetsWindow.Height - 3;
                int pages = packets.Count() / count;
                if ((packets.Count() % count) != 0)
                    pages++;
                return pages;
            }
        }

        /// <summary>
        /// Is Filtering Enabled
        /// </summary>
        public bool FilteringEnabled
        {
            get
            {
                return filteringEnabled;
            }
            set
            {
                filteringEnabled = value;
                UpdateFilteringStatus();
            }
        }

        /// <summary>
        /// Is Grouping Enabled
        /// </summary>
        public bool GroupingEnabled
        {
            get
            {
                return groupingEnabled;
            }
            set
            {
                groupingEnabled = value;
                UpdateGroupingStatus();
            }
        }
        #endregion



        /****************************************************************************************************/
        // :: PUBLIC METHODs ::
        /// <summary>
        /// Start PAIB Server
        /// </summary>
        public void Start()
        {
            if (!isGood)
                return;

            if (InitializeBrowser())
            {
                #region Browser Key Input Processing
                actualPage = 1;
                actualItem = 1;
                Refresh__PacketsWindow(actualItem);
                bool noUserExit = true;
                while (noUserExit)
                {
                    int maxItem = packets.Count() % maxCountPacketPerPage;
                    bool firstItem = true;

                    FrozeCursor();
                    ConsoleKeyInfo key = Console.ReadKey(); // listen input


                    #region Disable Forbidden Keys on Screen
                    FrozeCursor();
                    Console.Write(" ");
                    FrozeCursor();
                    #endregion

                    #region __Processing - Packets Window
                    //
                    // Pressing Down
                    //
                    if ((key.Key == ConsoleKey.DownArrow) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        actualItem++;
                        if ((actualItem > maxCountPacketPerPage) || ((actualItem > maxItem) && (maxItem != 0) && (actualPage == Pages)))
                        {
                            actualItem = 1;
                            actualPage++;
                            if (actualPage > Pages)
                                actualPage = 1;
                            Refresh__PacketsWindow();
                            Refresh__PacketsWindow(actualItem);
                        }
                        else
                            Refresh__PacketsWindow(actualItem);
                    }

                    //
                    // Pressing Up
                    //
                    else if ((key.Key == ConsoleKey.UpArrow) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        actualItem--;
                        if (actualItem == 0)
                        {
                            actualItem = maxCountPacketPerPage;
                            actualPage--;
                            if (actualPage == 0)
                            {
                                actualPage = Pages;
                                if (maxItem != 0)
                                    actualItem = maxItem;
                            }
                            Refresh__PacketsWindow();
                            Refresh__PacketsWindow(actualItem);
                        }
                        else
                            Refresh__PacketsWindow(actualItem);

                    }

                    //
                    // Pressing Right
                    //
                    else if ((key.Key == ConsoleKey.RightArrow) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        if (Pages == 1)
                        {
                        }
                        else
                        {
                            actualPage++;
                            if (actualPage > Pages)
                                actualPage = 1;
                            Refresh__PacketsWindow();
                            if ((actualPage == Pages) && (actualItem > maxItem) && (maxItem != 0))
                                actualItem = maxItem;
                            Refresh__PacketsWindow(actualItem);
                        }
                    }

                    //
                    // Pressing Left
                    //
                    else if ((key.Key == ConsoleKey.LeftArrow) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        if (Pages == 1)
                        {
                        }
                        else
                        {
                            actualPage--;
                            if (actualPage == 0)
                            {
                                actualPage = Pages;
                            }
                            Refresh__PacketsWindow();
                            if ((actualPage == Pages) && (actualItem > maxItem) && (maxItem != 0))
                                actualItem = maxItem;
                            Refresh__PacketsWindow(actualItem);
                        }
                    }

                    //
                    // Pressing Space
                    //
                    else if ((key.Key == ConsoleKey.Spacebar) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        Show_PacketsWindow(true, false);
                        Show_SearchWindow(true);
                    }

                    //
                    // Pressing Enter
                    //
                    else if ((key.Key == ConsoleKey.Enter) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        browserState = BrowserState.Info;
                        packetWindowIndex = ((actualPage - 1) * maxCountPacketPerPage + (actualItem - 1));
                        PacketWindow_Show(true, true, true);
                    }

                    //
                    // Pressing Escape
                    //
                    else if ((key.Key == ConsoleKey.Escape) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        noUserExit = false;
                    }

                    //
                    // Presing Help Button
                    //
                    else if ((key.Key == ConsoleKey.H) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        Show_HelpWindow(true);
                    }

                    //
                    // Pressing Grouping Button
                    //
                    else if ((key.Key == ConsoleKey.G) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        GetGroupingOnline();
                    }

                    //
                    // Pressing Filtering Button
                    //
                    else if ((key.Key == ConsoleKey.F) && (browserState == BrowserState.MainMenu) && (firstItem))
                    {
                        firstItem = false;
                        Show_FilterWindow(true);
                    }
                    #endregion

                    #region __Processing - Search Window
                    //
                    // Cancelling Search Window
                    //
                    if (((key.Key == ConsoleKey.Escape) || (key.Key == ConsoleKey.Spacebar)) && (browserState == BrowserState.Search) && (firstItem))
                    {
                        firstItem = false;
                        Show_SearchWindow(false);
                    }

                    //
                    // Typing Text
                    //
                    else if (((int)key.KeyChar >= (int)'0') && ((int)key.KeyChar <= (int)'9') && (browserState == BrowserState.Search) && (firstItem))
                    {
                        firstItem = false;
                        Console.BackgroundColor = searchWindow_Backcolor;
                        Console.ForegroundColor = searchWindow_Forecolor_typed;
                        Console.SetCursorPosition(searchWindowPosition, searchWindow.Y + 1);
                        Console.Write("{0}_", key.KeyChar);
                        searchWindowPosition++;
                        if (searchWindowPosition > searchWindowMaxPosition)
                        {
                            searchWindowPosition = searchWindowMaxPosition;
                            searchWindowExpression = searchWindowExpression.Substring(0, searchWindowExpression.Length - 1) + key.KeyChar;
                        }
                        else
                        {
                            searchWindowExpression += key.KeyChar;
                        }
                    }

                    //
                    // Backspacing Text
                    //
                    else if ((key.Key == ConsoleKey.Backspace) && (browserState == BrowserState.Search) && (firstItem))
                    {
                        firstItem = false;
                        Console.BackgroundColor = searchWindow_Backcolor;
                        Console.ForegroundColor = searchWindow_Forecolor_typed;
                        Console.SetCursorPosition(searchWindowPosition, searchWindow.Y + 1);
                        Console.Write("_ ", key.KeyChar);
                        searchWindowPosition--;
                        if (searchWindowPosition < searchWindowMinPosition)
                        {
                            searchWindowPosition = searchWindowMinPosition;
                            searchWindowExpression = "";
                        }
                        else
                        {
                            searchWindowExpression = searchWindowExpression.Substring(0, searchWindowExpression.Length - 1);
                        }
                    }

                    //
                    // Confirming Typed Texted
                    //
                    else if ((key.Key == ConsoleKey.Enter) && (browserState == BrowserState.Search) && (firstItem))
                    {
                        firstItem = false;
                        if (searchWindowExpression == "")
                        {
                            Show_SearchWindow(false);
                        }
                        else
                        {

                            bool bresult = false;
                            int value;
                            if (int.TryParse(searchWindowExpression, out value))
                            {
                                if ((value >= 1) && (value <= packets.Count()))
                                    bresult = true;
                            }

                            if (!bresult)
                            {
                                Show_SearchWindow_Errored(true);
                            }
                            else
                            {
                                Show_SearchWindow(false);
                                packetWindowIndex = ((actualPage - 1) * maxCountPacketPerPage + (actualItem - 1));
                                PacketWindow_Show(true, true, true);
                            }
                        }
                    }

                    //
                    // Packet not found!
                    //
                    else if (((key.Key == ConsoleKey.Enter) || (key.Key == ConsoleKey.Escape) || (key.Key == ConsoleKey.Spacebar)) && (browserState == BrowserState.Search_Error) && (firstItem))
                    {
                        firstItem = false;
                        Show_SearchWindow_Errored(false);
                    }
                    #endregion

                    #region __Processing - Filter Window
                    //
                    // Cancelling Search Window
                    //
                    if (((key.Key == ConsoleKey.Escape) || (key.Key == ConsoleKey.Spacebar)) && (browserState == BrowserState.Filter) && (firstItem))
                    {
                        firstItem = false;
                        Show_FilterWindow(false);
                    }

                    //
                    // Typing Text
                    //
                    else if (((int)key.KeyChar >= (int)'0') && ((int)key.KeyChar <= (int)'9') && (browserState == BrowserState.Filter) && (firstItem))
                    {
                        firstItem = false;
                        Console.BackgroundColor = searchWindow_Backcolor;
                        Console.ForegroundColor = searchWindow_Forecolor_typed;
                        Console.SetCursorPosition(filterWindowPosition, searchWindow.Y + 1);
                        Console.Write("{0}_", key.KeyChar);
                        filterWindowPosition++;
                        if (filterWindowPosition > filterWindowMaxPosition)
                        {
                            filterWindowPosition = filterWindowMaxPosition;
                            filterWindowExpression = filterWindowExpression.Substring(0, filterWindowExpression.Length - 1) + key.KeyChar;
                        }
                        else
                        {
                            filterWindowExpression += key.KeyChar;
                        }
                    }

                    //
                    // Backspacing Text
                    //
                    else if ((key.Key == ConsoleKey.Backspace) && (browserState == BrowserState.Filter) && (firstItem))
                    {
                        firstItem = false;
                        Console.BackgroundColor = searchWindow_Backcolor;
                        Console.ForegroundColor = searchWindow_Forecolor_typed;
                        Console.SetCursorPosition(filterWindowPosition, searchWindow.Y + 1);
                        Console.Write("_ ", key.KeyChar);
                        filterWindowPosition--;
                        if (filterWindowPosition < filterWindowMinPosition)
                        {
                            filterWindowPosition = filterWindowMinPosition;
                            filterWindowExpression = "";
                        }
                        else
                        {
                            filterWindowExpression = filterWindowExpression.Substring(0, filterWindowExpression.Length - 1);
                        }
                    }

                    //
                    // Confirming Typed Texted
                    //
                    else if ((key.Key == ConsoleKey.Enter) && (browserState == BrowserState.Filter) && (firstItem))
                    {
                        firstItem = false;
                        if (filterWindowExpression == "")
                        {
                            Show_FilterWindow(false);
                            filteringIndex = -1;
                            FilterPackets();
                        }
                        else
                        {

                            bool bresult = false;
                            if (int.TryParse(filterWindowExpression, out filteringIndex))
                            {
                                if ((filteringIndex >= 0) && (filteringIndex <= 255))
                                    bresult = true;
                            }

                            if (!bresult)
                            {
                                Show_FilterWindow_Errored(true);
                            }
                            else
                            {
                                Show_FilterWindow(false);
                                FilterPackets();
                            }
                        }
                    }

                    //
                    // Protocol not found!
                    //
                    else if (((key.Key == ConsoleKey.Enter) || (key.Key == ConsoleKey.Escape) || (key.Key == ConsoleKey.Spacebar)) && (browserState == BrowserState.Filter_Error) && (firstItem))
                    {
                        firstItem = false;
                        Show_FilterWindow_Errored(false);
                    }
                    #endregion

                    #region __Processing - Help Window
                    if (((key.Key == ConsoleKey.Enter) || (key.Key == ConsoleKey.Escape) || (key.Key == ConsoleKey.H)) && (browserState == BrowserState.Help) && (firstItem))
                    {
                        firstItem = false;
                        Show_HelpWindow(false);
                    }
                    #endregion

                    #region __Processing - Information Windows
                    #region __INNER_Processing - Packet Information Window
                    //
                    // Pressing Escape
                    //
                    if ((key.Key == ConsoleKey.Escape) && ((browserState == BrowserState.Info) || (browserState == BrowserState.Hex) || (browserState == BrowserState.Protocol)) && (firstItem))
                    {
                        firstItem = false;
                        PacketWindow_Show(false, false, true);
                    }

                    //
                    // Pressing Tab
                    //
                    else if ((key.Key == ConsoleKey.Tab) && (browserState == BrowserState.Info) && (firstItem))
                    {
                        firstItem = false;
                        PacketWindow_Show(false, true, false);
                        HexWindow_Show(true);
                    }

                    //
                    // Pressing DOWN or UP
                    //
                    else if (((key.Key == ConsoleKey.DownArrow) || (key.Key == ConsoleKey.UpArrow)) && (browserState == BrowserState.Info) && (firstItem))
                    {
                        firstItem = false;
                        int maxItems = packetWindow.Height - 1;
                        int newActual = 0;
                        if (key.Key == ConsoleKey.DownArrow)
                            newActual = packetWindow_Actual + 1;
                        if (key.Key == ConsoleKey.UpArrow)
                            newActual = packetWindow_Actual - 1;

                        if ((newActual > packetWindow_Data.Count) || (newActual < 1))
                        {

                        }
                        else
                        {
                            int newScrolling = 1 + newActual - maxItems;
                            if (newScrolling < 1)
                                newScrolling = 1;

                            if (newScrolling != packetWindow_Scrolling)
                            {
                                PacketWindow_RefreshBuffer(newActual, newScrolling, true, true);
                                packetWindow_Scrolling = newScrolling;
                            }
                            PacketWindow_RefreshItem(true, false, packetWindow_Actual, packetWindow_Scrolling);
                            PacketWindow_RefreshItem(true, true, newActual, packetWindow_Scrolling);
                            packetWindow_Actual = newActual;
                            HexWindow_RefreshItem(false, packetWindow_Actual);
                            HexWindow_RefreshBuffer(hexWindow_Scrolling, false, false);
                        }
                    }

                    //
                    // Pressing ENTER
                    //
                    else if ((key.Key == ConsoleKey.Enter) && (browserState == BrowserState.Info) && (firstItem))
                    {
                        firstItem = false;
                        if ((packetWindow_EnterConnentions.ContainsKey(packetWindow_Actual)) && (packetWindow_Actual != protocolWindow_Actual))
                        {
                            int newActual = packetWindow_EnterConnentions[packetWindow_Actual];
                            packetWindow_Actual = newActual;
                            PacketWindow_GetList(newActual);
                            PacketWindow_GetConnections(newActual);
                            HexWindow_GetList();
                            PacketWindow_RefreshBuffer(newActual, packetWindow_Scrolling, true, true);
                            PacketWindow_RefreshItem(true, true, packetWindow_Actual, packetWindow_Scrolling);

                            {

                                int maxItems = protocolWindow.Height - 1;
                                int newScrolling = protocolWindow_Scrolling;
                                if (newActual > maxItem)
                                {
                                    newScrolling = newActual - maxItems + 1;
                                }
                                if (newScrolling != protocolWindow_Scrolling)
                                {
                                    ProtocolWindow_RefreshBuffer(false, true, newScrolling);
                                    protocolWindow_Scrolling = newScrolling;
                                }
                                else
                                {
                                    ProtocolWindow_RefreshItem(false, false, protocolWindow_Actual);
                                    ProtocolWindow_RefreshItem(false, true, newActual);
                                }
                                protocolWindow_Actual = newActual;
                            }

                        }
                    }
                    #endregion

                    #region __INNER_Processing - Hex Window
                    //
                    // Pressing Tab
                    //
                    if ((key.Key == ConsoleKey.Tab) && (browserState == BrowserState.Hex) && (firstItem))
                    {
                        firstItem = false;
                        HexWindow_Show(false);
                        Show_ProtocolWindow(true);
                    }

                    //
                    // Pressing DOWN
                    //
                    else if ((key.Key == ConsoleKey.DownArrow) && (browserState == BrowserState.Hex) && (firstItem))
                    {
                        firstItem = false;
                        int maxItems = hexWindow.Height - 2;
                        int maxScrolling = 1 + hexWindow_Data.Count - maxItems;
                        if (maxScrolling < 1)
                            maxScrolling = 1;

                        int newScrolling = hexWindow_Scrolling + 1;
                        if (newScrolling > maxScrolling)
                        {
                        }
                        else
                        {
                            HexWindow_RefreshBuffer(newScrolling, true, true);
                            hexWindow_Scrolling = newScrolling;
                        }
                    }

                    //
                    // Pressing UP
                    //
                    else if ((key.Key == ConsoleKey.UpArrow) && (browserState == BrowserState.Hex) && (firstItem))
                    {
                        firstItem = false;
                        int maxItems = hexWindow.Height - 2;

                        int newScrolling = hexWindow_Scrolling - 1;
                        if (newScrolling < 1)
                        {
                        }
                        else
                        {
                            HexWindow_RefreshBuffer(newScrolling, true, true);
                            hexWindow_Scrolling = newScrolling;
                        }
                    }
                    #endregion

                    #region __INNER_Processing - Protocol Window
                    //
                    // Pressing Tab
                    //
                    if ((key.Key == ConsoleKey.Tab) && (browserState == BrowserState.Protocol) && (firstItem))
                    {
                        firstItem = false;
                        Show_ProtocolWindow(false);
                        PacketWindow_Show(false, true, true);
                    }

                    //
                    // Pressing DOWN or UP
                    //
                    else if (((key.Key == ConsoleKey.DownArrow) || (key.Key == ConsoleKey.UpArrow)) && (browserState == BrowserState.Protocol) && (firstItem))
                    {
                        firstItem = false;
                        int maxItems = protocolWindow.Height - 1;
                        int newActual = 0;

                        if (key.Key == ConsoleKey.DownArrow)
                            newActual = protocolWindow_Actual + 1;
                        if (key.Key == ConsoleKey.UpArrow)
                            newActual = protocolWindow_Actual - 1;

                        if ((newActual > packetData_Group.Count) || (newActual < 1))
                        {
                        }
                        else
                        {
                            packetWindow_Actual = newActual;
                            packetWindow_Scrolling = 1;
                            PacketWindow_GetList(newActual);
                            PacketWindow_GetConnections(newActual);
                            HexWindow_GetList();
                            PacketWindow_RefreshBuffer(newActual, packetWindow_Scrolling, true, false);
                            PacketWindow_RefreshItem(false, true, packetWindow_Actual, packetWindow_Scrolling);
                            HexWindow_RefreshItem(false, packetWindow_Actual);
                            HexWindow_RefreshBuffer(hexWindow_Scrolling, false, false);

                            int newScrolling = protocolWindow_Scrolling;
                            if (newActual > maxItem)
                            {
                                newScrolling = newActual - maxItems + 1;
                            }
                            if (newScrolling != protocolWindow_Scrolling)
                            {
                                ProtocolWindow_RefreshBuffer(true, true, newScrolling);
                                protocolWindow_Scrolling = newScrolling;
                            }
                            else
                            {
                                ProtocolWindow_RefreshItem(true, false, protocolWindow_Actual);
                                ProtocolWindow_RefreshItem(true, true, newActual);
                            }
                            protocolWindow_Actual = newActual;
                        }
                    }
                    #endregion
                    #endregion
                }
                #endregion
            }
            DeinitializeBrowser();
        }



        /****************************************************************************************************/
        // :: PRIVATE FUNCTIONs ::

        /// <summary>
        /// Get Packet Basic Information
        /// </summary>
        private void GetPacketBasicInfo()
        {
            {
                packetsNames_Active = new List<string>();
                packetsNames_Deactive = new List<string>();

                int index = 0;
                foreach (var packet in packets)
                {
                    index++;


                    string source = packet.IPSource;
                    string destination = packet.IPDestination;
                    string protocol = packet.Protocol;
                    string info = packet.Info;

                    source = source.ColoredSubstring(0, (source.GetLength() > 17) ? 17 : source.GetLength());
                    destination = destination.ColoredSubstring(0, (destination.GetLength() > 16) ? 16 : destination.GetLength());
                    protocol = protocol.ColoredSubstring(0, (protocol.GetLength() > 8) ? 8 : protocol.GetLength());
                    info = info.ColoredSubstring(0, (info.GetLength() > width - 59) ? width - 59 : info.GetLength());

                    if (info == "")
                    {
                        try
                        {
                            info = (packet.PortInfo.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries)).ToList()[0];
                            info = info.ColoredSubstring(0, (info.GetLength() > 21) ? 21 : info.GetLength());
                            info = "$08" + info;
                        }
                        catch
                        {
                        }
                    }


                    string emptyText = "";
                    for (int z = 1; z <= 80; z++)
                        emptyText += " ";
                    string deactiveText = String.Format("$07{0,3}.$08│ $15{1}", packetsNo[packet], source);
                    deactiveText += emptyText.Substring(0, 18 - source.GetLength());
                    deactiveText += String.Format("$08│ $15{0}", destination);
                    deactiveText += emptyText.Substring(0, 17 - destination.GetLength());
                    deactiveText += String.Format("$08│ $07{0}", protocol);
                    deactiveText += emptyText.Substring(0, 9 - protocol.GetLength());
                    deactiveText += String.Format("$08│ $15{0}", info);
                    deactiveText += emptyText.Substring(0, width - 58 - info.GetLength());

                    packetsNames_Deactive.Add(deactiveText);

                    string activeText = deactiveText.CleanColors();
                    packetsNames_Active.Add(activeText);

                }
            }
        }

        /// <summary>
        /// Make Grouping by Protocol No
        /// </summary>
        private void GetGroupingOnline()
        {
            string text = "Restoring packets grouping...";
            if (!GroupingEnabled)
                text = "Making packets grouping...";
            
            LoadingWindow_Show(true, text);
            {
                if (!GroupingEnabled)
                {
                    if (!FilteringEnabled)
                    {
                        packetsGrouping = packetsOriginal.OrderBy(packet => packet.Protocol.GetHashCode()).ToList();
                    }
                    else
                    {
                        packetsGrouping = packetsFiltered.OrderBy(packet => packet.Protocol.GetHashCode()).ToList();
                    }
                    packets = packetsGrouping;
                    GroupingEnabled = true;
                }
                else
                {
                    if (!FilteringEnabled)
                    {
                        packets = packetsOriginal;
                    }
                    else
                    {
                        packets = packetsFiltered;
                    }
                    GroupingEnabled = false;
                }

                GetPacketBasicInfo();
            }
            actualPage = 1;
            LoadingWindow_Show(false, "");
            Show_InfoWindow();
            Show_PacketsWindow(true, true);
            Refresh__PacketsWindow(actualItem);
        }

        /// <summary>
        /// Make Filtering by Protocol No
        /// </summary>
        /// <param name="portNo"></param>
        private void FilterPackets()
        {   
            string text = "Restoring packets filtering...";
            if (filteringIndex != -1)
                text = "Filtering packets...";

            LoadingWindow_Show(true, text);
            {
                if ((filteringIndex != -1) && (!FilteringEnabled))
                {
                    if (!GroupingEnabled)
                    {
                        packetsFiltered = packetsOriginal.Where(packet => packet.ProtocolNo == filteringIndex).ToList();
                    }
                    else
                    {
                        packetsFiltered = packetsGrouping.Where(packet => packet.ProtocolNo == filteringIndex).ToList();
                    }
                    packets = packetsFiltered;
                FilteringEnabled = true;
                }
                if (((filteringIndex == -1) && (FilteringEnabled)))
                {
                    if (!GroupingEnabled)
                    {
                            packets = packetsOriginal;
                    }
                    else
                    {
                            packets = packetsGrouping;
                    } 
                   FilteringEnabled = false;
                }
                GetPacketBasicInfo();
            }
            actualPage = 1;
            LoadingWindow_Show(false, "");
            Show_InfoWindow();
            Show_PacketsWindow(true, true);
            Refresh__PacketsWindow(actualItem);
        }

        

        #region __Graphics - Enable/Disable PBIG
        /// <summary>
        /// Initialize (PBIG) - Packet Browser Information Graphics
        /// </summary>
        private bool InitializeBrowser()
        {
            bool bResult = true;
            try
            {
                width = Console.BufferWidth - 1;
                height = Console.BufferHeight;
                if (width < 60)
                    bResult = false;
                if (height < 20)
                    bResult = false;
            }
            catch
            {
                width = 79;
                height = 25;
            }
            if (height > 25)
                height = 25;

            Show_Environment();

            packetsWindow = new Rectangle(0, 1, width - 1, height - 10);
            infoWindow = new Rectangle((width / 2) - 20, height - 7, 40, 4);
            searchWindow = new Rectangle((width / 2) - 13, (height / 2) + 1, 26, 2);

            helpWindow = new Rectangle((width / 2) - 19, (height / 2) - 4, 38, 10);

            packetWindow = new Rectangle(0, 1, width - 1, height - 11);
            hexWindow = new Rectangle(0, height - 9, width - 14, 7);
            protocolWindow = new Rectangle(width - 13, height - 9, 12, 7);

            LoadingWindow_Show(true, "Creating packets table...");
            GetPacketBasicInfo();
            LoadingWindow_Show(false, "");


            Show_InfoWindow();

            Show_PacketsWindow(true, true);
            return bResult;
        }

        /// <summary>
        /// Deinitialize (PBIG) - Packet Browser Information Graphics
        /// </summary>
        private void DeinitializeBrowser()
        {
            Console.BackgroundColor = ConsoleColor.Black;
            Console.ForegroundColor = ConsoleColor.Gray;
            //Console.ResetColor();
            Console.Clear();
        }
        #endregion

        #region __Graphics - Environment
        /// <summary>
        /// Set Graphics Environment
        /// </summary>
        private void Show_Environment()
        {
            Console.SetCursorPosition(0, 0);
            Console.BackgroundColor = ConsoleColor.Gray;
            Console.ForegroundColor = ConsoleColor.Black;
            Console.Clear();

            Console.BackgroundColor = ConsoleColor.DarkBlue;
            Console.ForegroundColor = ConsoleColor.Yellow;
            Console.SetCursorPosition(0, 0);
            for (int i = 1; i <= width; i++)
                Console.Write(" ");
            Console.SetCursorPosition(0, 0);
            Console.CursorLeft = (width / 2) - (AppSettings.AppName.Length / 2);
            Console.Write(AppSettings.AppName);

            Console.BackgroundColor = ConsoleColor.Black;
            Console.ForegroundColor = ConsoleColor.DarkGray;
            Console.SetCursorPosition(0, 1);
            for (int j = 1; j <= height - 2; j++)
            {
                for (int i = 1; i <= width; i++)
                    Console.Write("░");
                Console.WriteLine();
            }
            

            UpdateGroupingStatus();
            UpdateFilteringStatus();
            Status();
        }

        /// <summary>
        /// Set Cursor to Wait in the corner
        /// </summary>
        private void FrozeCursor()
        {
            Console.BackgroundColor = ConsoleColor.DarkBlue;
            Console.ForegroundColor = ConsoleColor.DarkBlue;
            Console.SetCursorPosition(0, 0);
        }

        /// <summary>
        /// Set Text Status
        /// </summary>
        /// <param name="message">message to be displayed</param>
        private void Status(string message)
        {
            Console.BackgroundColor = ConsoleColor.Gray;
            Console.ForegroundColor = ConsoleColor.Black;
            Console.SetCursorPosition(width - 1 - AppSettings.AppVer.Length, height - 1);
            
            Console.CursorLeft = 2;
            for (int i = 2; i <= width - 2; i++)
                Console.Write(" ");

            Console.SetCursorPosition(width - 1 - AppSettings.AppVer.Length, height - 1);
            Console.Write(AppSettings.AppVer);

            Console.CursorLeft = 2;
            ConsoleGraphics.ColoredWrite(message);
        }

        /// <summary>
        /// Set Actual Status to be displayed
        /// </summary>
        private void Status()
        {
            Status(statusMessages[browserState]);
        }

        private void UpdateFilteringStatus()
        {
            string text = "[Filtering]";
            Console.SetCursorPosition(width - text.Length, height - 2);
            if (GroupingEnabled)
                Console.SetCursorPosition(width - text.Length, height - 3);
            if (filteringEnabled)
            {
                Console.BackgroundColor = ConsoleColor.DarkGreen;
                Console.ForegroundColor = ConsoleColor.Green;
                Console.Write(text);
            }
            else
            {
                Console.BackgroundColor = ConsoleColor.Black;
                Console.ForegroundColor = ConsoleColor.DarkGray;
                for (int i = 1; i <= text.Length; i++)
                    Console.Write("░");
            }
            FrozeCursor();
        }

        private void UpdateGroupingStatus()
        {
            string text = "[Grouping]";
            Console.SetCursorPosition(width - text.Length, height - 2);
            if (groupingEnabled)
            {
                Console.BackgroundColor = ConsoleColor.DarkCyan;
                Console.ForegroundColor = ConsoleColor.Cyan;
                Console.Write(text);
            }
            else
            {
                Console.BackgroundColor = ConsoleColor.Black;
                Console.ForegroundColor = ConsoleColor.DarkGray;
                for (int i = 1; i <= text.Length; i++)
                    Console.Write("░");
            }
            FrozeCursor();
        }

        /// <summary>
        /// Construct Console Graphics Window
        /// </summary>
        /// <param name="title">window title</param>
        /// <param name="area">window area coordinates</param>
        /// <param name="forecolor">window forecolor</param>
        /// <param name="backcolor">window background</param>
        private void MakeWindow(string title, Rectangle area, ConsoleColor forecolor, ConsoleColor backcolor)
        {
            MakeWindow(title, area, forecolor, backcolor, true);
        }

        /// <summary>
        /// Construct Console Graphics Window
        /// </summary>
        /// <param name="title">window title</param>
        /// <param name="area">window area coordinates</param>
        /// <param name="forecolor">window forecolor</param>
        /// <param name="backcolor">window background</param>
        /// <param name="doubled">frame is double lined</param>
        private void MakeWindow(string title, Rectangle area, ConsoleColor forecolor, ConsoleColor backcolor, bool doubled)
        {

            Console.BackgroundColor = backcolor;
            Console.ForegroundColor = forecolor;


            Console.SetCursorPosition(area.X, area.Y);
            if (doubled)
                Console.Write("╔");
            else
                Console.Write("┌");
            for (int i = area.X + 1; i <= area.X + area.Width - 1; i++)
            {
                if (doubled)
                    Console.Write("═");
                else
                    Console.Write("─");
            }
            if (doubled)
                Console.Write("╗");
            else
                Console.Write("┐");

            for (int j = area.Y + 1; j <= area.Y + area.Height - 1; j++)
            {
                Console.SetCursorPosition(area.X, j);
                if (doubled)
                    Console.Write("║");
                else
                    Console.Write("│");

                for (int i = area.X + 1; i <= area.X + area.Width - 1; i++)
                {
                    Console.Write(" ");
                }
                if (doubled)
                    Console.Write("║");
                else
                    Console.Write("│");
            }

            Console.SetCursorPosition(area.X, area.Y + area.Height);
            if (doubled)
                Console.Write("╚");
            else
                Console.Write("└");
            for (int i = area.X + 1; i <= area.X + area.Width - 1; i++)
            {
                if (doubled)
                    Console.Write("═");
                else
                    Console.Write("─");
            }
            if (doubled)
                Console.Write("╝");
            else
                Console.Write("┘");


            if (doubled)
            {
                Console.SetCursorPosition(area.X + 2, area.Y);
                Console.Write("[ ]");

                Console.SetCursorPosition(area.X + 3, area.Y);
                Console.ForegroundColor = ConsoleColor.Green;
                Console.Write("■");
            }

            Console.ForegroundColor = forecolor;
            Console.SetCursorPosition((area.Width / 2) - ((title.Length + 2) / 2) + area.X, area.Y);
            Console.Write(String.Format(" {0} ", title));

        }
        #endregion

        #region __Graphics - Basic Information Window
        /// <summary>
        /// Refresh Info Window
        /// </summary>
        private void Refresh__InfoWindow()
        {
            #region Fill Info Window
            Console.SetCursorPosition(infoWindow.X + 2, infoWindow.Y + 1);
            string name = packager.OriginalName;
            if (name.Length > 32)
                name = name.Substring(0, 32);
            ConsoleGraphics.ColoredWrite(String.Format("$15File: $14{0}", name));

            Console.SetCursorPosition(infoWindow.X + 2, infoWindow.Y + 2);
            string size = packager.Size.ByteSize();
            if (size.Length > 32)
                size = size.Substring(0, 32);
            ConsoleGraphics.ColoredWrite(String.Format("$15Size: $14{0}", size));

            Console.SetCursorPosition(infoWindow.X + 2, infoWindow.Y + 3);
            string count = packets.Count().ToString();
            if (count.Length > 21)
                count = count.Substring(0, 21);
            ConsoleGraphics.ColoredWrite(String.Format("$15Total packets: $14{0}", count));
            #endregion
        }

        /// <summary>
        /// Show Basic Information Window
        /// </summary>
        private void Show_InfoWindow()
        {
            MakeWindow("Trace File Information", infoWindow, ConsoleColor.Black, ConsoleColor.Gray, false);
            Refresh__InfoWindow();
            Status();
        }
        #endregion

        #region __Graphics - Packets Window
        /// <summary>
        /// Refresh Packets Information Window
        /// </summary>
        private void Refresh__PacketsWindow()
        {
            Refresh__PacketsWindow(true);
        }

        /// <summary>
        /// Refresh Packets Information Window
        /// </summary>
        /// <param name="active">window state active</param>
        private void Refresh__PacketsWindow(bool active)
        {
            #region Clean Window
            if (active)
            {
                Console.ForegroundColor = packetsWindow_Forecolor;
                Console.BackgroundColor = packetsWindow_Backcolor;
            }
            else
            {
                Console.ForegroundColor = window_Inactive_Forecolor;
                Console.BackgroundColor = window_Inactive_Backcolor;
            }
            for (int j = packetsWindow.Y + 3; j <= packetsWindow.Y + packetsWindow.Height - 1; j++)
            {
                Console.SetCursorPosition(packetsWindow.X + 1, j);
                for (int i = packetsWindow.X + 1; i <= packetsWindow.X + packetsWindow.Width - 1; i++)
                    Console.Write(" ");
            }
            #endregion

            #region Table Header
            if (active)
            {
                Console.BackgroundColor = packetsWindow_Backcolor;
                Console.ForegroundColor = packetsWindow_Forecolor;
            }
            else
            {
                Console.ForegroundColor = window_Inactive_Forecolor;
                Console.BackgroundColor = window_Inactive_Backcolor;
            }
            Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 1);
            for (int j = packetsWindow.Y + 1; j <= packetsWindow.Y + packetsWindow.Height - 1; j++)
            {
                Console.SetCursorPosition(packetsWindow.X + 1, j);

                Console.CursorLeft = packetsWindow.X + 5;
                Console.Write("│");
                Console.CursorLeft = packetsWindow.X + 25;
                Console.Write("│");
                Console.CursorLeft = packetsWindow.X + 44;
                Console.Write("│");
                Console.CursorLeft = packetsWindow.X + 55;
                Console.Write("│");
            }

            if (active)
                Console.ForegroundColor = packetsWindow_Forecolor_HighLight;
            else
                Console.ForegroundColor = window_Inactive_Forecolor;
            Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 1);
            Console.Write(" No");
            Console.CursorLeft = packetsWindow.X + 7;
            Console.Write("Source");
            Console.CursorLeft = packetsWindow.X + 27;
            Console.Write("Destination");
            Console.CursorLeft = packetsWindow.X + 46;
            Console.Write("Protocol");
            Console.CursorLeft = packetsWindow.X + 57;
            Console.Write("Information");

            if (active)
                Console.ForegroundColor = packetsWindow_Forecolor;
            else
                Console.ForegroundColor = window_Inactive_Forecolor;
            Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 2);
            for (int i = packetsWindow.X + 1; i <= packetsWindow.X + packetsWindow.Width - 1; i++)
            {
                Console.Write("─");
            }
            Console.CursorLeft = packetsWindow.X + 5;
            Console.Write("┼");
            Console.CursorLeft = packetsWindow.X + 25;
            Console.Write("┼");
            Console.CursorLeft = packetsWindow.X + 44;
            Console.Write("┼");
            Console.CursorLeft = packetsWindow.X + 55;
            Console.Write("┼");
            #endregion

            #region Table Content
            maxCountPacketPerPage = packetsWindow.Height - 3; // default is 12
            int leftFilter = (actualPage - 1) * maxCountPacketPerPage;
            IEnumerable<Packet> viewPackets = packets.Where(item => packets.ToList().IndexOf(item) >= leftFilter);
            if (viewPackets.Count() == 0)
            {
                return; //errored
            }
            viewPackets = viewPackets.Take(maxCountPacketPerPage);

            int relativeIndex = 0;
            foreach (var packet in viewPackets)
            {
                Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 3 + relativeIndex);

                if (active)
                    ConsoleGraphics.ColoredWrite(String.Format("{0}", packetsNames_Deactive[(actualPage - 1) * maxCountPacketPerPage + relativeIndex]));
                else
                    Console.Write(String.Format("{0}", packetsNames_Active[(actualPage - 1) * maxCountPacketPerPage + relativeIndex]));
                relativeIndex++;
            }
            #endregion

            Show_PacketsWindow_ActualPage(active);
        }

        /// <summary>
        /// Activate/Deactivate Selected Packet
        /// </summary>
        /// <param name="index"></param>
        private void Refresh__PacketsWindow(int index)
        {
            int maxCountPacketPerPage = packetsWindow.Height - 3;

            //
            // Activate given line 
            //
            Console.BackgroundColor = packetsWindow_Backcolor_HighLight;
            Console.ForegroundColor = packetsWindow_Forecolor;
            Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 3 + index - 1);
            try
            {
                ConsoleGraphics.ColoredWrite(String.Format("{0}", packetsNames_Active[(actualPage - 1) * maxCountPacketPerPage + index - 1]));
            }
            catch
            {
            }

            //
            // Deactivate last possible lines
            //
            int previndex = index - 1;
            if (previndex == 0)
                previndex = maxCountPacketPerPage;
            int nextindex = index + 1;
            if (nextindex == maxCountPacketPerPage + 1)
                nextindex = 1;
            Console.BackgroundColor = packetsWindow_Backcolor;
            Console.ForegroundColor = packetsWindow_Forecolor;
            try
            {
                Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 3 + previndex - 1);
                int prevIndexNo = (actualPage - 1) * maxCountPacketPerPage + previndex - 1;
                ConsoleGraphics.ColoredWrite(String.Format("{0}", packetsNames_Deactive[prevIndexNo]));
            }
            catch
            {
            }
            try
            {
                Console.SetCursorPosition(packetsWindow.X + 1, packetsWindow.Y + 3 + nextindex - 1);
                int nextIndexNo = (actualPage - 1) * maxCountPacketPerPage + nextindex - 1;
                ConsoleGraphics.ColoredWrite(String.Format("{0}", packetsNames_Deactive[nextIndexNo]));
            }
            catch
            {
            }
        }

        /// <summary>
        /// Enable/Disable Packets Window
        /// </summary>
        /// <param name="enabled">enable/disable switch</param>
        /// <param name="active">active/deactive switch</param>
        private void Show_PacketsWindow(bool enabled, bool active)
        {
            if (enabled)
            {
                if (active)
                {
                    MakeWindow("Packets", packetsWindow, packetsWindow_Forecolor, packetsWindow_Backcolor);
                    browserState = BrowserState.MainMenu;
                    Status();
                }
                else
                    MakeWindow("Packets", packetsWindow, window_Inactive_Forecolor, window_Inactive_Backcolor);
                Refresh__PacketsWindow(active);
            }
            else
            {

            }
        }

        /// <summary>
        /// Show Actual Page Information
        /// </summary>
        private void Show_PacketsWindow_ActualPage(bool active)
        {
            if (active)
            {
                Console.ForegroundColor = packetsWindow_Forecolor;
                Console.BackgroundColor = packetsWindow_Backcolor;
            }
            else
            {
                Console.ForegroundColor = window_Inactive_Forecolor;
                Console.BackgroundColor = window_Inactive_Backcolor;
            }

            string emptyString = "════════════════════";
            string pageString = String.Format("[Page {0} / {1}]", actualPage, Pages);

            Console.SetCursorPosition(packetsWindow.X + packetsWindow.Width - emptyString.Length - 1, packetsWindow.Y + packetsWindow.Height);
            Console.Write(emptyString);
            Console.SetCursorPosition(packetsWindow.X + packetsWindow.Width - pageString.Length - 1, packetsWindow.Y + packetsWindow.Height);
            Console.Write(pageString);
        }
        #endregion

        #region __Graphics - Search Window
        /// <summary>
        /// Enable/Disable Search Window
        /// </summary>
        /// <param name="enable">enable/disable switch</param>
        private void Show_SearchWindow(bool enabled)
        {
            if (enabled)
            {
                MakeWindow("Search", searchWindow, searchWindow_Forecolor, searchWindow_Backcolor, false);
                Console.ForegroundColor = searchWindow_Forecolor_texted;
                Console.SetCursorPosition(searchWindow.X + 1, searchWindow.Y + 1);

                string typedText = " Type packet no: ";
                Console.Write(typedText);

                searchWindowMinPosition = Console.CursorLeft;
                searchWindowMaxPosition = searchWindowMinPosition + searchWindow.Width - typedText.Length - 3;
                searchWindowPosition = searchWindowMinPosition;
                

                Console.ForegroundColor = searchWindow_Forecolor_typed;
                Console.CursorLeft = searchWindowMinPosition;
                Console.Write("_");

                searchWindowExpression = "";

                FrozeCursor();
                browserState = BrowserState.Search;
                Status();
            }
            else
            {
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow();
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
                Status();
            }
        }

        /// <summary>
        /// Enable/Disable Search Error Window
        /// </summary>
        /// <param name="enabled"></param>
        private void Show_SearchWindow_Errored(bool enabled)
        {
            if (enabled)
            {
                string typedText = "Cannot find such packet!";
                MakeWindow("Warning", new Rectangle((width / 2) - (typedText.Length / 2) - 1, searchWindow.Y, typedText.Length + 3, searchWindow.Height), searchWindow_Forecolor, ConsoleColor.DarkRed, false);
                Console.ForegroundColor = searchWindow_Forecolor_texted;
                Console.SetCursorPosition((width / 2) - (typedText.Length / 2) + 1, searchWindow.Y + 1);
                Console.Write(typedText);

                FrozeCursor();
                browserState = BrowserState.Search_Error;
                Status();
            }
            else
            {
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow();
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
                Status();
            }
        }
        #endregion

        #region __Graphics - Filter Window
        /// <summary>
        /// Enable/Disable Search Window
        /// </summary>
        /// <param name="enable">enable/disable switch</param>
        private void Show_FilterWindow(bool enabled)
        {
            if (enabled)
            {
                MakeWindow("Filter", searchWindow, searchWindow_Forecolor, searchWindow_Backcolor, false);
                Console.ForegroundColor = searchWindow_Forecolor_texted;
                Console.SetCursorPosition(searchWindow.X + 1, searchWindow.Y + 1);

                string typedText = " Type protocol no: ";
                Console.Write(typedText);

                filterWindowMinPosition = Console.CursorLeft;
                filterWindowMaxPosition = filterWindowMinPosition + searchWindow.Width - typedText.Length - 3;
                filterWindowPosition = filterWindowMinPosition;


                Console.ForegroundColor = searchWindow_Forecolor_typed;
                Console.CursorLeft = filterWindowMinPosition;
                Console.Write("_");

                filterWindowExpression = "";

                FrozeCursor();
                browserState = BrowserState.Filter;
                Status();
            }
            else
            {
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow();
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
                Status();
            }
        }

        /// <summary>
        /// Enable/Disable Search Error Window
        /// </summary>
        /// <param name="enabled"></param>
        private void Show_FilterWindow_Errored(bool enabled)
        {
            if (enabled)
            {
                string typedText = "Bad protocol no! (0-255)";
                MakeWindow("Warning", new Rectangle((width / 2) - (typedText.Length / 2) - 1, searchWindow.Y, typedText.Length + 3, searchWindow.Height), searchWindow_Forecolor, ConsoleColor.DarkRed, false);
                Console.ForegroundColor = searchWindow_Forecolor_texted;
                Console.SetCursorPosition((width / 2) - (typedText.Length / 2) + 1, searchWindow.Y + 1);
                Console.Write(typedText);

                FrozeCursor();
                browserState = BrowserState.Filter_Error;
                Status();
            }
            else
            {
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow();
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
                Status();
            }
        }
        #endregion

        #region __Graphics - Help Window
        /// <summary>
        /// Enable/Disable Help Window
        /// </summary>
        /// <param name="enabled">enable/disable switch</param>
        private void Show_HelpWindow(bool enabled)
        {
            if (enabled)
            {
                MakeWindow("Controls Help", helpWindow, helpWindow_Forecolor, helpWindow_Backcolor);
                #region Graphics Style Settings
                string hT = ConsoleGraphics.consoleColors[helpWindow_Forecolor_texted].ToString();
                if (hT.Length == 1)
                    hT = "0" + hT;
                hT = "$" + hT;
                string hH = ConsoleGraphics.consoleColors[helpWindow_Forecolor_HighLight].ToString();
                if (hH.Length == 1)
                    hH = "0" + hH;
                hH = "$" + hH;
                string line = "";
                for (int i = 1; i <= helpWindow.Width - 1; i++)
                    line += "─";
                #endregion

                #region Filling Help Window
                int index = helpWindow.Y + 1;
                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}▲/▼ {1}Browsing packets", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}◄/► {1}Browsing pages", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}└─> {1}Select", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 1, index++);
                ConsoleGraphics.ColoredWrite(String.Format("$08{0}",line));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}SPACE - {1}Choose single packet", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}G - {1}Grouping packets", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}F - {1}Filter packets", hH, hT));

                Console.SetCursorPosition(helpWindow.X + 1, index++);
                ConsoleGraphics.ColoredWrite(String.Format("$08{0}", line));

                Console.SetCursorPosition(helpWindow.X + 2, index++);
                ConsoleGraphics.ColoredWrite(String.Format("{0}ESC - {1}Exit Packet Browser", hH, hT));
                #endregion

                browserState = BrowserState.Help;
                Status();
            }
            else
            {
                Show_Environment();
                Show_InfoWindow();
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
            }
        }
        #endregion

        #region __Graphics - Loading Window
        /// <summary>
        /// Enable/Disable Loading Window
        /// </summary>
        /// <param name="enabled">enable/disable switch</param>
        /// <param name="message">window text</param>
        private void LoadingWindow_Show(bool enabled, string message)
        {
            if (enabled)
            {
                MakeWindow("Loading", new Rectangle((width / 2) - (message.Length / 2) - 2, (height / 2) - 1, message.Length + 5, 2), ConsoleColor.Black, ConsoleColor.Gray, false);
                Console.ForegroundColor = ConsoleColor.White;
                Console.SetCursorPosition((width / 2) - (message.Length / 2) + 1, (height / 2));
                Console.Write(message);

                browserState = BrowserState.Loading;
                Status();
                FrozeCursor();
            }
            else
            {
                Show_Environment();
            }
        }
        #endregion

        #region __Graphics - Information Windows
        #region __INNER_Graphics - PacketWindow
        /// <summary>
        /// Create "packetWindow_Data" collection & "packetWindow_EnterConnection" collection from "packetData_Info" object
        /// </summary>
        /// <param name="seed">protocol index no for extending</param>
        private void PacketWindow_GetList(int seed)
        {
            packetWindow_Data = new List<string>();
            packetWindow_EnterConnentions = new Dictionary<int, int>();
            int maxWidth = packetWindow.Width - 1;

            int actualSeed = 0;
            int actualPos = 0;
            foreach (var info in packetData_Info)
            {
                actualSeed++;

                if ((actualSeed != seed) || ((actualSeed == seed) && (info.Value == null)))
                {
                    actualPos++;
                    packetWindow_EnterConnentions[actualPos] = actualSeed;
                    packetWindow_Data.Add(String.Format("$15+ {0}", info.Key.ColoredSubstring(0, (info.Key.Length > maxWidth - 2) ? maxWidth - 2 : info.Key.Length)));
                }
                else
                {
                    actualPos++;
                    packetWindow_EnterConnentions[actualPos] = actualSeed;
                    packetWindow_Data.Add(String.Format("$14- {0}", info.Key.ColoredSubstring(0, (info.Key.Length > maxWidth - 2) ? maxWidth - 2 : info.Key.Length)));
                    IDictionary<string, object> items = (IDictionary<string, object>)info.Value;

                    foreach (var item in items)
                    {
                        if (item.Value == null)
                        {
                            actualPos++;
                            packetWindow_Data.Add(String.Format("$07    {0}", item.Key.ColoredSubstring(0, (item.Key.Length > maxWidth - 4) ? maxWidth - 4 : item.Key.Length)));
                        }
                        else
                        {
                            actualPos++;
                            packetWindow_Data.Add(String.Format("$07  - {0}", item.Key.ColoredSubstring(0, (item.Key.Length > maxWidth - 4) ? maxWidth - 4 : item.Key.Length)));
                            IDictionary<string, object> innerItems = (IDictionary<string, object>)item.Value;
                            foreach (var innerItem in innerItems)
                            {
                                actualPos++;
                                packetWindow_Data.Add(String.Format("$08      {0}", innerItem.Key.ColoredSubstring(0, (innerItem.Key.Length > maxWidth - 6) ? maxWidth - 6 : innerItem.Key.Length)));
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Create "packetWindow_HexConnection" collection from "packetData_Hex" object
        /// </summary>
        /// <param name="seed">protocol index no for extending</param>
        private void PacketWindow_GetConnections(int seed)
        {
            packetWindow_HexConnections = new List<Point>();

            int index = 0;
            foreach (var hexLevel1 in packetData_Hex)
            {
                index++;

                packetWindow_HexConnections.Add(hexLevel1.Key);

                if ((hexLevel1.Value != null) && (index == seed))
                {
                    IList<KeyValuePair<Point, object>> hexLevel1Collection = (IList<KeyValuePair<Point, object>>)hexLevel1.Value;
                    foreach (var hexLevel2 in hexLevel1Collection)
                    {
                        packetWindow_HexConnections.Add(hexLevel2.Key);

                        if (hexLevel2.Value != null)
                        {
                            IList<KeyValuePair<Point, object>> hexLevel2Collection = (IList<KeyValuePair<Point, object>>)hexLevel2.Value;
                            foreach (var hexLevel3 in hexLevel2Collection)
                            {
                                packetWindow_HexConnections.Add(hexLevel3.Key);
                            }
                        }
                    }
                }
            }

        }

        /// <summary>
        /// Visualize "packetWindow_Data" content in packetWindow buffer
        /// </summary>
        /// <param name="seed">protocol no for extending</param>
        /// <param name="scrolling">scrolling position (1-?)</param>
        /// <param name="clear">clearing switch</param>
        /// <param name="active">window activating switch</param>
        private void PacketWindow_RefreshBuffer(int seed, int scrolling, bool clear, bool active)
        {
            if (active)
            {
                Console.BackgroundColor = packetWindow_Backcolor;
                Console.ForegroundColor = packetWindow_Forecolor;
            }
            else
            {
                Console.BackgroundColor = window_Inactive_Backcolor;
                Console.ForegroundColor = window_Inactive_Forecolor;
            }

            #region CLEARING WINDOW
            if (clear)
            {
                for (int j = packetWindow.Y + 1; j <= packetWindow.Y + packetWindow.Height - 1; j++)
                {
                    Console.SetCursorPosition(packetWindow.X + 1, j);
                    for (int i = packetWindow.X + 1; i <= packetWindow.X + packetWindow.Width - 1; i++)
                    {
                        Console.Write(" ");
                    }
                }
            }
            #endregion

            if (scrolling < 1)
                scrolling = 1;
            int listing = 0;
            listing = packetWindow_Data.Count + 1 - scrolling;

            if (listing != 0)
            {
                int maxItems = packetWindow.Height - 1;
                if (listing > maxItems)
                    listing = maxItems;
                for (int i = 1; i <= listing; i++)
                {
                    Console.SetCursorPosition(packetWindow.X + 1, packetWindow.Y + i);
                    if (active)
                        ConsoleGraphics.ColoredWrite(String.Format("{0}", packetWindow_Data[scrolling + i - 2]));
                    else
                    {
                        ConsoleGraphics.ColoredWrite(String.Format("$08{0}", packetWindow_Data[scrolling + i -2].CleanColors()));
                    }
                }

            }
            
        }

        /// <summary>
        /// Visualize "packetWindow_Data" one item in packetWindow buffer
        /// </summary>
        /// <param name="active">window activate switch</param>
        /// <param name="isHighlight">item is highlighted</param>
        /// <param name="itemToHighlight">item no in packetData_info to make highlighted</param>
        /// <param name="scrolling">scrolling position</param>
        private void PacketWindow_RefreshItem(bool active, bool isHighlight, int itemToHighlight, int scrolling)
        {
            int maxItems = packetWindow.Height - 1;

            int maxWidth = packetWindow.Width - 1;
            string empty = "";
            for (int a = 1; a <= 80; a++)
                empty += " ";
            string highlightText = packetWindow_Data[itemToHighlight - 1];

            
            int positionY = itemToHighlight;
            if (positionY > maxItems)
                positionY = maxItems;
            Console.SetCursorPosition(packetWindow.X + 1, packetWindow.Y + positionY);
            if (isHighlight)
            {
                highlightText = String.Format("{0}", highlightText.CleanColors() + empty.ColoredSubstring(0, maxWidth - highlightText.GetLength()));
                if (active)
                {
                    Console.BackgroundColor = ConsoleColor.DarkCyan;
                    ConsoleGraphics.ColoredWrite(String.Format("$15{0}", highlightText));
                }
                else
                {
                    Console.BackgroundColor = ConsoleColor.DarkCyan;
                    ConsoleGraphics.ColoredWrite(String.Format("$00{0}", highlightText));
                }
            }
            else
            {
                if (active)
                    Console.BackgroundColor = packetWindow_Backcolor;
                else
                    Console.BackgroundColor = window_Inactive_Backcolor;
                highlightText = highlightText + empty.ColoredSubstring(0, maxWidth - highlightText.GetLength());
                ConsoleGraphics.ColoredWrite(highlightText);
            }
            
        }

        /// <summary>
        /// Enable/Disable Packet Window
        /// </summary>
        /// <param name="firstRun">is first run</param>
        /// <param name="enabled">enable/disable switch</param>
        /// <param name="active">active/deactive switch</param>
        private void PacketWindow_Show(bool firstRun, bool enabled, bool active)
        {
            if (enabled)
            {
                if (firstRun)
                {
                    Show_Environment();
                    LoadingWindow_Show(true, "Preparing packet information for viewing...");
                    int index = ((actualPage - 1) * maxCountPacketPerPage) + (actualItem - 1);
                    Packet choosedPacket = packets.ToList()[index];
                    IList<object> packetData = (IList<object>)choosedPacket.GetFullInformation(); //Packet More Information Loading;
                    packetData_Info = (IDictionary<string, object>)packetData[0];
                    packetData_Hex = (IList<KeyValuePair<Point, object>>)packetData[1];
                    packetData_Group = (IList<string>)packetData[2];
                    packetData_Data = (byte[])packetData[3];

                    protocolWindow_Actual = packetData_Group.Count;
                    protocolWindow_Scrolling = 1;

                    hexWindow_Scrolling = 1;

                    packetWindow_Actual = protocolWindow_Actual;
                    packetWindow_Scrolling = 1;

                    PacketWindow_GetList(protocolWindow_Actual);
                    PacketWindow_GetConnections(protocolWindow_Actual);
                    HexWindow_GetList();

                    LoadingWindow_Show(false, "");
                }

                if (active)
                {
                    MakeWindow("Packet Information", packetWindow, packetWindow_Forecolor, packetWindow_Backcolor);
                    PacketWindow_RefreshBuffer(protocolWindow_Actual, protocolWindow_Scrolling, true, true);
                    PacketWindow_RefreshItem(true, true, packetWindow_Actual, packetWindow_Scrolling);
                }
                else
                {
                    MakeWindow("Packet Information", packetWindow, window_Inactive_Forecolor, window_Inactive_Backcolor);
                    PacketWindow_RefreshBuffer(protocolWindow_Actual, protocolWindow_Scrolling, true, false);
                    PacketWindow_RefreshItem(false, true, packetWindow_Actual, packetWindow_Scrolling);
                }

                
                

                HexWindow_Show(false);
                Show_ProtocolWindow(false);
                browserState = BrowserState.Info;
                Status();
            }
            else
            {
                packetData_Data = null;
                packetData_Group = null;
                packetData_Hex = null;
                packetData_Info = null;
                packetWindow_Data = null;
                packetWindow_HexConnections = null;
                packetWindow_EnterConnentions = null;
                hexWindow_Data = null;
                packetWindow_Actual = 0;
                protocolWindow_Actual = 0;
                packetWindow_Scrolling = 0;
                hexWindow_Scrolling = 0;
                protocolWindow_Scrolling = 0;
                Show_Environment();
                Show_InfoWindow();
                Show_PacketsWindow(true, true);
                Refresh__PacketsWindow(actualItem);
                browserState = BrowserState.MainMenu;
                Status();
            }
        }
        #endregion

        #region __INNER_Graphics - HexWindow
        /// <summary>
        /// Create "windowHex_Data" collection from "packetData_Data" byte object
        /// </summary>
        private void HexWindow_GetList()
        {
            int length = packetData_Data.Length;

            int verticalItems = length / 16;
            int verticalItemsRest = length % 16;
            if (verticalItemsRest != 0)
                verticalItems++;

            hexWindow_Data = new List<string>();

            
            Stream stream = new MemoryStream(packetData_Data);
            stream.Position = 0;
            int index = 0;
            for (int i = 1; i <= verticalItems; i++)
            {
                byte[] byteline = new byte[16];
                int readed = stream.Read(byteline, 0, 16);

                int extrahigh = index / 256;
                int high = (index % 256) / 16;
                int low = index % 16;

                string line = String.Format("{0:X}{1:X}{2:X}0 ", extrahigh, high, low);

                string hexView = "";
                string charView = "";
                for (int k = 1; k <= 16; k++)
                {
                    string hexViewNew = String.Format("{0}", byteline[k-1].ConvertToOneHex());
                    if ((i == verticalItems) && (k > verticalItemsRest))
                        hexViewNew = "  ";
                    hexView += hexViewNew;
                    if ((k % 2) == 0)
                        hexView += " ";
                    if ((k % 8) == 0)
                        hexView += " ";

                    if (k == 9)
                        charView += " ";
                    char ch = (char)byteline[k - 1];
                    int num = (int)ch;
                    if (((num >= 7) && (num < 26)) || (num == 27) || (num == 36) || (num == 183)) // controlling chars
                        ch = '.';
                    else
                    {
                        ch = (char)Char.ConvertToUtf32((char)0xD800, (char)(0xDC00 | ch));
                    }


                    string chS = String.Format("{0}", ch);
                    if (chS.Length == 0)
                        chS = " ";

                    if ((i == verticalItems) && (k > verticalItemsRest))
                        chS = " ";
                    charView += chS;
                    
                }

                line = line + hexView + charView;
                hexWindow_Data.Add(line);
                index++;
            }

            

        }


        /// <summary>
        /// Visualize "windowHex_Data" content in hexWindow buffer
        /// </summary>
        /// <param name="scrolling">scrolling position</param>
        /// <param name="clear">clearing switch</param>
        /// <param name="active">window activating switch</param>
        private void HexWindow_RefreshBuffer(int scrolling, bool clear, bool active)
        {
            
            if (active)
            {
                Console.BackgroundColor = hexWindow_Backcolor;
                Console.ForegroundColor = hexWindow_Forecolor;
            }
            else
            {
                Console.BackgroundColor = window_Inactive_Backcolor;
                Console.ForegroundColor = window_Inactive_Forecolor;
            }

            #region CLEARING WINDOW
            if (clear)
            {
               
                for (int j = hexWindow.Y + 1; j <= hexWindow.Y + hexWindow.Height - 1; j++)
                {
                    Console.SetCursorPosition(hexWindow.X + 1, j);
                    for (int i = hexWindow.X + 1; i <= hexWindow.X + hexWindow.Width - 1; i++)
                    {
                        Console.Write(" ");
                    }
                }
            }
            #endregion

            int listing = 0;
            listing = hexWindow_Data.Count + 1 - scrolling;

            Console.SetCursorPosition(hexWindow.X + 1, hexWindow.Y + 1);
            string header = "     0001 0203 0405 0607  0809 0A0B 0C0D 0EEF  01234567 89ABCDEF";
            if (active)
                header = "$03" + header;
            else
                header = "$08" + header;
            ConsoleGraphics.ColoredWrite(String.Format("{0}", header));

            if (listing != 0)
            {
                int maxItems = hexWindow.Height - 2;
                if (listing > maxItems)
                    listing = maxItems;
                for (int i = 1; i <= listing; i++)
                {
                    if (active)
                    {
                        Console.BackgroundColor = hexWindow_Backcolor;
                        Console.ForegroundColor = hexWindow_Forecolor;
                    }
                    else
                    {
                        Console.BackgroundColor = window_Inactive_Backcolor;
                        Console.ForegroundColor = window_Inactive_Forecolor;
                    }

                    Console.SetCursorPosition(hexWindow.X + 1, hexWindow.Y + i + 1);
                    string textedLine = hexWindow_Data[scrolling + i - 2];
                    string prefix = textedLine.Substring(0, 4);
                    textedLine = textedLine.Substring(4, textedLine.Length - 4);
                    if (active)
                        ConsoleGraphics.ColoredWrite(String.Format("$03{0}$15{1}", prefix, textedLine));
                    else
                    {
                        ConsoleGraphics.ColoredWrite(String.Format("$08{0}$00{1}", prefix, textedLine));
                    }
                }

            }


        }

        /// <summary>
        /// Visualize "windowHex_Data" item's span in hexWindow buffer
        /// </summary>
        /// <param name="active">window activating switch</param>
        /// <param name="actualPosition">actual "packetWindow" position</param>
        private void HexWindow_RefreshItem(bool active, int actualPosition)
        {
            #region __CLEANING "hexWindow_Data"
            HexWindow_GetList();
            #endregion

            Point span = packetWindow_HexConnections[actualPosition - 1];
            HexWindow_RefreshItem_Inner(active, true, span);
        }

        /// <summary>
        /// NON USER FUNC - Visualize GIVEN "windowHex_Data" item in hexWindow buffer with specialized color
        /// </summary>
        /// <param name="position">position when start coloring</param>
        /// <param name="replacedColor">color on string</param>
        /// <param name="replacedColorOff">color off string</param>
        private void HexWindow_RefreshItem_Inner(int position, string replacedColor, string replacedColorOff, string replacedBackColor, string replacedBackColorOff)
        {
            int value = position;

            int linePos = (value - 1) / 16;
            string line = hexWindow_Data[linePos];
            int charPos = (value - 1) % 16 + 1;




            int insertPosition = 5 + 2 * (charPos - 1);
            insertPosition += (charPos - 1) / 2;
            if (charPos > 8)
                insertPosition += 1;

            string lineLeft = line.ColoredSubstring(0, insertPosition);
            string lineMiddle = line.ColoredSubstring(lineLeft.Length, 47 - insertPosition);
            string lineRight = line.ColoredSubstring(lineLeft.Length + lineMiddle.Length, line.GetLength() - 47);
            {
                int insertPosition2 = (charPos) - 1;
                if (charPos > 8)
                    insertPosition2++;
                string lineRightLeft = lineRight.ColoredSubstring(0, insertPosition2);
                string lineRightRight = lineRight.Substring(lineRightLeft.Length, lineRight.Length - lineRightLeft.Length);
                lineRight = lineRightLeft + replacedColor + lineRightRight;
            }
            line = lineLeft + replacedColor + lineMiddle + replacedColorOff + lineRight;

            hexWindow_Data[linePos] = line;
        }

        /// <summary>
        /// NON USER FUNC - Visualize "windowHex_Data" item's span in hexWindow buffer (Run inner functions to operate)
        /// </summary>
        /// <param name="active"></param>
        /// <param name="span"></param>
        private void HexWindow_RefreshItem_Inner(bool active, bool switcher, Point span)
        {
            #region Defined Colors

            string activeoff = (ConsoleGraphics.consoleColors[hexWindow_Forecolor]).ToString();
            string nonactiveoff = (ConsoleGraphics.consoleColors[window_Inactive_Forecolor]).ToString();
            if (activeoff.Length == 1)
                activeoff = "0" + activeoff;
            activeoff = "$" + activeoff;
            if (nonactiveoff.Length == 1)
                nonactiveoff = "0" + nonactiveoff;
            nonactiveoff = "$" + nonactiveoff;

            string on = "$14";
            string activeoffBack = "$" + (ConsoleGraphics.consoleColors[hexWindow_Backcolor] + 16).ToString();
            string nonactiveoffBack = "$" + (ConsoleGraphics.consoleColors[window_Inactive_Backcolor] + 16).ToString();
            string onBack = "$" + (ConsoleGraphics.consoleColors[ConsoleColor.Blue] + 16).ToString();
            string offBack = nonactiveoffBack;
            if (active)
                offBack = activeoffBack;

            string replaceStringOn = on;
            string replaceStringOff = nonactiveoff;
            if (active)
                replaceStringOff = activeoff;

            string replaceString = replaceStringOff;
            if (switcher)
                replaceString = replaceStringOn;
            #endregion


            int left = (span.X - 1) / 16;
            int right = (span.Y - 1) / 16;
            for (int z = left; z <= right; z++)
            {
                if (z == left)
                {
                    HexWindow_RefreshItem_Inner(span.X, replaceStringOn, replaceStringOff, onBack, offBack);
                }
                else if (z == right)
                {
                    HexWindow_RefreshItem_Inner(z * 16 + 1, replaceStringOn, replaceStringOff, onBack, offBack);
                }
                else
                {
                    HexWindow_RefreshItem_Inner(z * 16 + 1, replaceStringOn, replaceStringOff, onBack, offBack);
                }
            }

            int leftBlank = span.X - 1;
            if (leftBlank < 1)
                leftBlank = 0;
            if (leftBlank != 0)
                HexWindow_RefreshItem_Inner(leftBlank, replaceStringOff, replaceStringOff, onBack, offBack);
            int rightBlank = span.Y + 1;
            if (rightBlank > packetData_Data.Length)
                rightBlank = 0;
            if (rightBlank != 0)
                HexWindow_RefreshItem_Inner(rightBlank, replaceStringOff, replaceStringOff, onBack, offBack);


            
        }

       


        /// <summary>
        /// Enable/Disable Hex Window
        /// </summary>
        /// <param name="active">active/deactive switch</param>
        private void HexWindow_Show(bool active)
        {
            if (active)
            {
                MakeWindow("Hex Viewer", hexWindow, hexWindow_Forecolor, hexWindow_Backcolor, false);
            }
            else
            {
                MakeWindow("Hex Viewer", hexWindow, window_Inactive_Forecolor, window_Inactive_Backcolor, false);
            }
            HexWindow_RefreshItem(active, packetWindow_Actual);
            HexWindow_RefreshBuffer(hexWindow_Scrolling, false, active);
            browserState = BrowserState.Hex;
            Status();
        }
        #endregion

        #region __INNER_Graphics - ProtocolWindow
        /// <summary>
        /// Visualize "packetData_Group" content into "protocolWindow" buffer
        /// </summary>
        /// <param name="active">window activating switch</param>
        /// <param name="clear">clearing switch</param>
        /// <param name="scrolling">scrolling position</param>
        private void ProtocolWindow_RefreshBuffer(bool active, bool clear, int scrolling)
        {

            if (active)
            {
                Console.BackgroundColor = protocolWindow_Backcolor;
                Console.ForegroundColor = protocolWindow_Forecolor;
            }
            else
            {
                Console.BackgroundColor = window_Inactive_Backcolor;
                Console.ForegroundColor = window_Inactive_Forecolor;
            }
            #region CLEARING WINDOW
            if (clear)
            {

                for (int j = protocolWindow.Y + 1; j <= protocolWindow.Y + protocolWindow.Height - 1; j++)
                {
                    Console.SetCursorPosition(protocolWindow.X + 1, j);
                    for (int i = protocolWindow.X + 1; i <= protocolWindow.X + protocolWindow.Width - 1; i++)
                    {
                        Console.Write(" ");
                    }
                }
            }
            #endregion

            if (scrolling < 1)
                scrolling = 1;
            int listing = 0;
            listing = packetData_Group.Count + 1 - scrolling;

            if (listing != 0)
            {
                int maxItems = protocolWindow.Height - 1;
                if (listing > maxItems)
                    listing = maxItems;
                for (int i = 1; i <= listing; i++)
                {

                    Console.SetCursorPosition(protocolWindow.X + 2, protocolWindow.Y + i);
                    string textedLine = packetData_Group[scrolling + i - 2];
                   
                    if (active)
                        ConsoleGraphics.ColoredWrite(String.Format("$07{0}", textedLine));
                    else
                    {
                        ConsoleGraphics.ColoredWrite(String.Format("$08{0}", textedLine));
                    }
                }

            }

        }

        /// <summary>
        /// Visualize "packetData_Group" given item into "protocolWindow" buffer
        /// </summary>
        /// <param name="ative"></param>
        /// <param name="seed"></param>
        private void ProtocolWindow_RefreshItem(bool active, bool isHighlight, int itemToHighlight)
        {
            int maxItems = protocolWindow.Height - 1;

            int maxWidth = protocolWindow.Width - 1 - 2;
            string empty = "";
            for (int a = 1; a <= 80; a++)
                empty += " ";
            string highlightText = packetData_Group[itemToHighlight - 1];


            int positionY = itemToHighlight;
            if (positionY > maxItems)
                positionY = maxItems;
            Console.SetCursorPosition(protocolWindow.X + 1, protocolWindow.Y + positionY);
            if (isHighlight)
            {
                highlightText = String.Format("{0}", highlightText.CleanColors() + empty.ColoredSubstring(0, maxWidth - highlightText.GetLength()));
                if (active)
                {
                    Console.BackgroundColor = ConsoleColor.Yellow;
                    ConsoleGraphics.ColoredWrite(String.Format("$08 {0} ", highlightText));
                }
                else
                {
                    Console.BackgroundColor = ConsoleColor.Yellow;
                    ConsoleGraphics.ColoredWrite(String.Format("$08 {0} ", highlightText));
                }
            }
            else
            {
                if (active)
                {
                    Console.BackgroundColor = protocolWindow_Backcolor;
                    highlightText = "$07 " + highlightText + empty.ColoredSubstring(0, maxWidth - highlightText.GetLength()) + " ";
                }
                else
                {
                    Console.BackgroundColor = window_Inactive_Backcolor;
                    highlightText = "$08 " + highlightText + empty.ColoredSubstring(0, maxWidth - highlightText.GetLength()) + " ";
                }
                ConsoleGraphics.ColoredWrite(highlightText);
            }
        }

        /// <summary>
        /// Enable/Disable Protocol Window
        /// </summary>
        /// <param name="active">active/deactive switch</param>
        private void Show_ProtocolWindow(bool active)
        {
            if (active)
            {
                MakeWindow("Content", protocolWindow, protocolWindow_Forecolor, protocolWindow_Backcolor, false);
            }
            else
            {
                MakeWindow("Content", protocolWindow, window_Inactive_Forecolor, window_Inactive_Backcolor, false);
            }
            ProtocolWindow_RefreshBuffer(active, true, protocolWindow_Scrolling);
            ProtocolWindow_RefreshItem(active, true, protocolWindow_Actual);
            browserState = BrowserState.Protocol;
            Status();
        }
        #endregion
        #endregion


    }
}
