﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using PacMap.General;
using System.Drawing;

namespace PacMap.Protocols
{
    /// <summary>
    /// User Datagram Protocol
    /// </summary>
    public class UDP : Protocol
    {
        /// <summary>
        /// Constructs UDP Class
        /// </summary>
        /// <param name="input">bytes array</param>
        /// <param name="owner">packet frame</param>
        public UDP(byte[] input, Packet.Packet owner) : base(input, owner)
        {
            shortName = "UDP";
            Stream stream = new MemoryStream(input);

            //
            // Get Source Port
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                SourcePort = (word[0] << 8 | word[1]);
            }

            //
            // Get Destination Port
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                DestinationPort = (word[0] << 8 | word[1]);
            }

            //
            // Get Length
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                length = (word[0] << 8 | word[1]);
            }

            //
            // Get Checksum
            //
            {
                byte[] word = new byte[2];
                stream.Read(word, 0, 2);
                hChecksum = (word[0] << 8 | word[1]);
            }


            //
            // Verify Header - not ready
            //
            //int headerCheck = (~((sourcePort + destinationPort + length) & 0xFFFF) & 0xFFFF);


            //
            // Additionals Information exporting to packet
            //
            packet.LocalPort = Port;
            packet.RemotePort = RemotePort;
            {
                mainInformation = String.Format("$07{0} > {1}", SourcePort.GetSinglePortName(ProtocolType.UDP), DestinationPort.GetSinglePortName(ProtocolType.UDP));
                packet.Info = mainInformation;
            }
            //--------------------------------------------
            int restSize = (int)(stream.Length - stream.Position);
            byte[] rest = new byte[restSize];
            int readed = stream.Read(rest, 0, restSize);

            if (readed == 0)
                return;


            child = ProtocolsInfo.CreateAppropriateProtocolContent(ProtocolType.UDP, Port, RemotePort, packet, rest);

        }


        /***********************************************************************************************************/
        // :: DATA FIELDs ::
        private int localPort = 0x0000;
        private int remotePort = 0x0000;
        private int length = 0x0000;
        private int hChecksum = 0x0000;
        
        /***********************************************************************************************************/
        // :: PROPERTIEs ::
        #region PROPERTIEs
        /// <summary>
        /// Get or Set Local Port
        /// </summary>
        public int Port
        {
            get
            {
                return localPort;
            }
            set
            {
                localPort = value;
            }
        }

        /// <summary>
        /// Get or Set Remote Port
        /// </summary>
        public int RemotePort
        {
            get
            {
                return remotePort;
            }
            set
            {
                remotePort = value;
            }
        }

        /// <summary>
        /// Get or Set Source Port
        /// </summary>
        public int SourcePort
        {
            get
            {
                if (packet.Direction == Packet.Direction.LocalToRemote)
                    return localPort;
                else
                    return remotePort;
            }
            set
            {
                if (packet.Direction == Packet.Direction.LocalToRemote)
                    localPort = value;
                else
                    remotePort = value;
            }
        }

        /// <summary>
        /// Get or Set Destination Port
        /// </summary>
        public int DestinationPort
        {
            get
            {
                if (packet.Direction == Packet.Direction.RemoteToLocal)
                    return localPort;
                else
                    return remotePort;
            }
            set
            {
                if (packet.Direction == Packet.Direction.RemoteToLocal)
                    localPort = value;
                else
                    remotePort = value;
            }
        }

        /// <summary>
        /// Get UDP Length
        /// </summary>
        public int Length
        {
            get
            {
                return length;
            }
        }
        #endregion


        /***********************************************************************************************************/
        // :: GLOBAL METHODs ::


        ///// <summary>
        ///// Save Protocol to bytes array
        ///// </summary>
        ///// <returns>bytes array</returns>
        //public override byte[] Save()
        //{
        //    return base.Save();
        //}


        ///// <summary>
        ///// Start AnonMapping Protocol
        ///// </summary>
        //public override void Anonymize()
        //{
        //    base.Anonymize();
        //}


        /// <summary>
        /// Get Full Packet Information for Advanced Packet Viewing
        /// </summary>
        /// <returns>Specialized Information Object</returns>
        public override object GetFullInformation()
        {
            IList<object> result = new List<object>();
            IDictionary<string, object> browserInfo = new Dictionary<string, object>();
            IList<KeyValuePair<Point, object>> browserHex = new List<KeyValuePair<Point, object>>();
            IList<string> browserGroup = new List<string>();
            result.Add(browserInfo);
            result.Add(browserHex);
            result.Add(browserGroup);

            #region __GET NAMEs
            {
                string source = String.Format("{0} ({1})", (SourcePort.GetPortName(ProtocolType.UDP) == "") ? SourcePort.ToString() : SourcePort.GetPortName(ProtocolType.UDP), SourcePort);
                string destination = String.Format("{0} ({1})", (DestinationPort.GetPortName(ProtocolType.UDP) == "") ? DestinationPort.ToString() : DestinationPort.GetPortName(ProtocolType.UDP), DestinationPort);
                string name = String.Format("User Datagram Protocol, Src Port: {0}, Dst Port: {1}", source, destination);
                string item1 = String.Format("Source port: {0}", source);
                string item2 = String.Format("Destination port: {0}", destination);
                string item3 = String.Format("Length: {0}", Length);
                string checksumString = "";
                string checksumResult = "[Verification Not Implemented Yet]";
                {
                    int left = (hChecksum & 0xFF00) >> 8;
                    int right = (hChecksum & 0x00FF);
                    string leftString = String.Format("{0:X}", left);
                    if (leftString.Length == 1)
                        leftString = "0" + leftString;
                    string rightString = String.Format("{0:X}", right);
                    if (rightString.Length == 1)
                        rightString = "0" + rightString;
                    checksumString = "0x" + leftString + rightString;
                }
                string item4 = String.Format("Checksum: {0} {1}", checksumString, checksumResult);
                IDictionary<string, object> inner = new Dictionary<string, object>();
                inner[item1] = null;
                inner[item2] = null;
                inner[item3] = null;
                inner[item4] = null;
                browserInfo[name] = inner;
            }
            #endregion

            #region __GET HEX
            {
                Point nameHex = new Point(1, 8);
                Point item1Hex = new Point(1, 2);
                Point item2Hex = new Point(3, 4);
                Point item3Hex = new Point(5, 6);
                Point item4Hex = new Point(7, 8);
                IList<KeyValuePair<Point, object>> innerHex = new List<KeyValuePair<Point, object>>();
                innerHex.Add(new KeyValuePair<Point, object>(item1Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item2Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item3Hex, null));
                innerHex.Add(new KeyValuePair<Point, object>(item4Hex, null));
                browserHex.Add(new KeyValuePair<Point, object>(nameHex, innerHex));
            }
            #endregion

            #region __GET GROUP
            {
                browserGroup.Add(shortName);
            }
            #endregion

            return result;
        }
        



    }
}
